(ns yegortimoshenko.pdf
  (:refer-clojure :exclude [merge])
  (:require [clojure.java.io :as io])
  (:import [com.itextpdf.io.font FontConstants]
           [com.itextpdf.kernel.color Color]
           [com.itextpdf.kernel.events Event IEventHandler PdfDocumentEvent]
           [com.itextpdf.kernel.font PdfFontFactory]
           [com.itextpdf.kernel.pdf PdfDocument PdfReader PdfWriter]
           [com.itextpdf.kernel.pdf.canvas PdfCanvas]
           [com.itextpdf.kernel.utils PdfMerger]
           [com.itextpdf.layout Document]
           [com.itextpdf.layout.element AreaBreak Paragraph]))

(def extension "pdf")
(def mime-type "application/pdf")

(defn merge [source target]
  (with-open [w (PdfMerger. (PdfDocument. (PdfWriter. (io/output-stream target))))]
    (doseq [s source]
      (with-open [r (PdfDocument. (PdfReader. (io/input-stream s)))]
        (.merge w r 1 (.getNumberOfPages r))))))

(def event-types
  {:end PdfDocumentEvent/END_PAGE
   :insert PdfDocumentEvent/INSERT_PAGE
   :remove PdfDocumentEvent/REMOVE_PAGE
   :start PdfDocumentEvent/START_PAGE})

(defn handle [doc t f]
  (.addEventHandler ^PdfDocument doc (t event-types)
    (reify IEventHandler
      (^void handleEvent [this ^Event e] (f e)))))

(defn watermark [{:keys [color font size text] :or {color Color/BLACK font FontConstants/HELVETICA size 12 :text ""}} ^PdfCanvas canvas x y]
  (doto canvas
    (.beginText)
    (.setFontAndSize (PdfFontFactory/createFont font) size)
    (.setFillColor color)
    (.moveText x y)
    (.showText ^String text)
    (.endText)
    (.release)))

(defn footer [options source target]
  (with-open [pdf (PdfDocument. (PdfReader. (io/input-stream source)) (PdfWriter. (io/output-stream target)))]
    (handle pdf :end
      (fn [e]
        (let [doc (Document. pdf)]
          (watermark options (PdfCanvas. (.getPage e)) (.getLeftMargin doc) (.getBottomMargin doc)))))))

(defn pagination
  ([source target] (pagination {} source target))
  ([{:keys [start end] :or {start 1} :as options} source target]
   (let [counter (atom 0)]
     (with-open [pdf (PdfDocument. (PdfReader. (io/input-stream source)) (PdfWriter. (io/output-stream target)))]
       (handle pdf :end
               (fn [e]
                 (let [doc (Document. pdf)
                       page (.getPage e)
                       size (.getPageSize page)
                       no (.getPageNumber pdf page)]
                   (when (and (>= no start) (if end (< no end) true))
                     (watermark (assoc options :text (str (swap! counter inc)))
                                (PdfCanvas. page)
                                (- (.getRight size) (.getRightMargin doc))
                                (.getBottomMargin doc))))))))))
