(ns yegortimoshenko.docx
  (:refer-clojure :exclude [format])
  (:require [clojure.java.io :as io]
            [clojure.string :as str])
  (:import [pl.jsolve.templ4docx.core Docx VariablePattern]
           [pl.jsolve.templ4docx.variable ImageVariable TextVariable Variables]
           [org.docx4j Docx4J]
           [org.docx4j.openpackaging.packages WordprocessingMLPackage]
           [java.io File]
           [javax.imageio ImageIO]))

(defprotocol IPDF
  (-available? [this])
  (-convert [this source target]))

(defrecord XSL-FO []
  IPDF
  (-available? [this] true)
  (-convert [this source target]
    (let [doc (WordprocessingMLPackage/load (io/file source))
          settings (doto (Docx4J/createFOSettings) (.setWmlPackage doc))]
      (with-open [out (io/output-stream (io/file target))]
        (Docx4J/toFO settings out Docx4J/FLAG_EXPORT_PREFER_XSL)))))

;; I plan to add documents4j and JODConverter support in the future and use
;; them when available (see the according method in the IPDF protocol).
(defn ->pdf
  ([source target] (->pdf (XSL-FO.) source target))
  ([impl source target] (-convert impl source target)))

;; Compatible with mail merge templates.
(def ^:dynamic *prefix* "«")
(def ^:dynamic *suffix* "»")

(defn pattern ^String [s]
  (apply str (interpose (name s) [*prefix* *suffix*])))

(defn format [fmt ctx]
  (reduce-kv (fn [s k v] (str/replace s (pattern k) v)) fmt ctx))

(defprotocol IVariable
  (-add [this k vars]))

(defrecord Image [^File f ^int w ^int h]
  IVariable
  (-add [this k vars] (.addImageVariable ^Variables vars (ImageVariable. (pattern k) f w h))))

(extend File IVariable
  {:-add (fn [^File this k vars]
           (let [image (ImageIO/read this)]
             (.addImageVariable ^Variables vars (ImageVariable. (pattern k) this (.getWidth image) (.getHeight image)))))})

(extend String IVariable
  {:-add (fn [^String this k vars]
           (.addTextVariable ^Variables vars (TextVariable. (pattern k) this)))})

(defn render [ctx source target]
  (let [docx (Docx. source)
        vars (Variables.)]
    (.setVariablePattern docx (VariablePattern. *prefix* *suffix*))
    (doseq [[k v] ctx] (-add v k vars))
    (.fillTemplate docx vars)
    (with-open [out (io/output-stream target)]
      (.save docx out))))
