(ns com.wsscode.log
  (:require
    [clojure.string :as str]
    [com.fulcrologic.guardrails.core :refer [<- => >def >defn >fdef ? |]])
  #?(:cljs
     (:require-macros
       [com.wsscode.log]))
  #?(:clj
     (:import
       (java.util
         Date))))

(>def ::level #{::level-debug
                ::level-info
                ::level-warn
                ::level-error})

(def log-levels
  {::level-debug 1
   ::level-info  2
   ::level-warn  3
   ::level-error 4})

(defn level-name [level]
  (str/upper-case (subs (name level) 6)))

(defn level-unqualified-kw [level]
  (keyword (subs (name level) 6)))

(defn print-logger [_]
  (fn print-logger-internal [{::keys [timestamp level event] :as data}]
    (print (str
             timestamp " "
             (if level (str (level-name level) " "))
             event
             " - "
             (pr-str (dissoc data ::level ::event ::timestamp ::line ::column))
             "\n"))))

(def ^:dynamic *active-logger* (print-logger {::min-level ::level-debug}))

(defn now []
  #?(:clj  (Date.)
     :cljs (js/Date.)))

(defn make-event [event-level event-name data]
  (merge
    {::event     event-name
     ::level     event-level
     ::timestamp (now)}
    data))

(defn event-macro-data [form]
  (let [{:keys [line column]} (meta form)]
    {::line      line
     ::column    column
     ::file      *file*
     ::namespace (quote (symbol (str *ns*)))}))

(defn capture [level form event-name event-data]
  (let [base-data (event-macro-data form)]
    `(*active-logger*
       (make-event ~level ~event-name ~(merge base-data event-data)))))

#?(:clj
   (defmacro debug
     [event-name event-data]
     (capture ::level-debug &form event-name event-data)))

#?(:clj
   (defmacro info
     [event-name event-data]
     (capture ::level-info &form event-name event-data)))

#?(:clj
   (defmacro warn
     [event-name event-data]
     (capture ::level-warn &form event-name event-data)))

#?(:clj
   (defmacro error
     [event-name event-data]
     (capture ::level-error &form event-name event-data)))

#?(:clj
   (defmacro with-logger [logger & body]
     `(binding [*active-logger* ~logger]
        ~@body)))
