(ns com.wsscode.pathom-docs.getting-started
  (:require [com.wsscode.pathom.core :as p]))

; for simplicity, let's use an "atom database"
(def contacts
  {:ww {:contact/id          :ww
        :contact/name        "Walter White"
        :contact/friends-ids [:jp]}
   :jp {:contact/id          :jp
        :contact/name        "Jessy Pinkman"
        :contact/friends-ids []}})

; the contacts is an atom, but could be your db instance or a database connection,
; whatever works to fetch your data
(defn load-contact [{::keys [contacts]} id]
  (get contacts id))

; See dispatch helpers docs for more information on entity-dispatch
(defmulti entity-reader p/entity-dispatch)

; load the contact entity by id, eg: [:contact/id :ww]
(defmethod entity-reader :contact/id [env]
  (let [id (p/ident-value env)]
    ; load contact and join to process the subquery
    (p/join (load-contact env id) env)))

; make sure to provide a default continuation for multi-methods to allow reader
; composition
(defmethod entity-reader :default [_] ::p/continue)

; See dispatch helpers docs for more information on key-dispatch
(defmulti virtual-attr p/key-dispatch)

; Here we use the virtual-attr to define a global access since it has no
; requirements on the current entity it can be called from anywhere in the graph
(defmethod virtual-attr :contacts/all [{::keys [contacts] :as env}]
  ; this time we use the join-seq to walk parsing a sequence
  (p/join-seq env (vals contacts)))

; To load a relationship, we are going to use the current entity
; :contact/friends-ids to fetch the entities
(defmethod virtual-attr :contact/friends [env]
  ; p/entity returns the entity that is currently being processed, giving as access
  ; to it's properties
  (let [{:contact/keys [friends-ids]} (p/entity! env [:contact/friends-ids])]
    (->> friends-ids
         ; transform each id in a contact
         (map (partial load-contact env))
         ; join to parse the sub-query
         (p/join-seq env))))

; again, allow for composition
(defmethod virtual-attr :default [_] ::p/continue)

; use p/parser to init the parser
(def parser
  ; the env plugin will inject enviroment variables on the parser at the start of
  ; the run
  (p/parser {::p/plugins [(p/env-plugin {::p/reader [p/map-reader virtual-attr entity-reader]})
                          ; error handling plugin, catch exceptions during parsing,
                          ; allowing for the parser to keep running even when an
                          ; exception happens, returning whatever worked
                          p/error-handler-plugin]}))

(comment
  ; Play around with the parser
  (parser {::contacts contacts}
          [{[:contact/id :ww] [:contact/friends]}
           {:contacts/all [:contact/name]}]))
