(ns wonka.core
  (:require ["glamor" :as glamor]
            [cljs.spec.alpha :as s]
            [goog :refer [typeOf]]
            [goog.memoize]
            [oops.core :refer [ocall]]
            [wonka.reset :refer [meyer-reset]]))

(def CONFIG (atom nil))

(def ^:private dash-rgx (js* "/-(.)/g"))

(defn- upcase [_ s]
  (.toUpperCase s))

(defn- camelize-style-name*
  "Camelcases a hyphenated CSS property name, for example:

   > camelizeStyleName('background-color')
   < \"backgroundColor\"
   > camelizeStyleName('-moz-transition')
   < \"MozTransition\"
   > camelizeStyleName('-ms-transition')
   < \"msTransition\""
  [style-name]
  (.replace (name style-name) dash-rgx upcase))

(def ^:private camelize-style-name (goog.memoize camelize-style-name*))

(defn get-theme-var [v]
  (get (:theme @CONFIG) v {}))

(defn get-theme-prefix []
  (get @CONFIG :theme-prefix ""))

(defn theme-rgx []
  (re-pattern (str "^" (get-theme-prefix) "\\.?")))

(defn theme-kw? [k]
  (and (keyword? k)
       (re-find (theme-rgx) (or (namespace k) ""))))

(defn resolve-theme-kw [v]
  (let [resolved-value (get-theme-var v)]
    (when (nil? resolved-value)
      (.log js/console "Could not resolve css rule '" v "'."))
    resolved-value))

(defn- resolve-rule-values
  "Recurs down the rule and resolves keywords values in the styleguide."
  [rule]
  (persistent!
    (reduce-kv
      (fn [acc k v]
        (let [rule-key (if (keyword? k)
                         (camelize-style-name k)
                         k)
              rule-val (cond
                         ;; Special paper keywords
                         (theme-kw? v)
                         (resolve-theme-kw v)

                         ;; Nested maps
                         (map? v)
                         (resolve-rule-values v)

                         :else
                         v)]
          (assoc! acc rule-key rule-val)))
      (transient {})
      rule)))

(defn- add-label [rule label]
  (update rule :label (fnil identity label)))

(defn css-rule
  "Creates and returns a new glamor rule from a CLJ map. Used internally by the
  `let-css` macro."
  ^js [rule label]
  (-> rule
      resolve-rule-values
      (add-label label)
      clj->js
      glamor/css))

(defn glamor? [rule]
  (= (typeOf rule) "object"))

(s/def ::glamor? (s/and glamor?))

(defn css

  "Takes a variable number of arguments, which could be:
  - A map of CSS properties
  - A glamor object generated by a previous `css` call
  - A CSS class name as a string or keyword
  - A `nil` value (so we can use `when`s)

  and returns a glamor object or a coll that includes a glamor object and any
  CSS class names as strings."
  [& rules]
  {:pre  [(s/coll-of string? keyword? map? nil? ::glamor?)]
   :post [(s/or :g ::glamor? :c coll? :n nil?)]}
  (when-not (empty? rules)
    (let [grouped   (group-by
                      #(cond
                         (or (string? %) (keyword? %)) :static
                         (or (map? %) (glamor? %)) :processable
                         :else :drop)
                      (flatten rules))
          processed (->> (:processable grouped)
                         (reduce
                           (fn [acc rule]
                             (if (map? rule)
                               (conj! acc (css-rule rule nil))
                               (conj! acc rule)))
                           (transient []))
                         persistent!
                         (apply glamor/css))]
      (if (empty? (:static grouped))
        processed
        (->> (:static grouped)
             (map name)
             dedupe
             (concat [processed]))))))

(defn keyframes
  "Creates and returns a glamor keyframe definition. Use it inside css:

  (def kf (keyframes {:from {:opacity 0}
                      :to   {:opacity 1}}))

  (let-css [class-a {:animation kf}]
    [:div {:class class-a}])"
  ^js [timeline]
  (ocall glamor/css
         :keyframes
         (gensym "paper_keyframes")
         (-> timeline
             resolve-rule-values
             clj->js)))

(defn insert-global! [selector rule]
  (ocall glamor/css
         :global
         (name selector)
         (-> rule resolve-rule-values clj->js)))

(defn join
  "CSS allows you to specify many shorthand properties as a single string
  (ex: `background-image: url(images/bg.gif); background-repeat: no-repeat;`
  becomes `background: url(images/bg.gif) no-repeat`). String concat isn't fun
  in Clojure, so you can use the paper.core/join to a sequence of properties
  into CSS shorthand string. Automatically appends \"px\" to integers and does
  paper namespaced keyword lookups in the styleguide.

  Ex:

  (let-css [x {:background (p/join \"url(images/bg.gif)\" \"no-repeat\")}]
    [:div {:class x}])"
  [& properties]
  (->> properties
       (map (fn [x]
              (cond
                (integer? x)  (str x "px")
                (theme-kw? x) (resolve-theme-kw x)
                (keyword? x)  (name x)
                :else         x)))
       (interpose " ")
       (apply str)))

(defn config [{:keys [theme-prefix theme] :as config}]
  (reset! CONFIG config)
  config)

(defn insert! [css-str]
  (ocall glamor/css :insert css-str))

(defn insert-reset! []
  (insert! meyer-reset))
