(ns com.webcomrades.horza.xsd
  (:require [clojure.data.xml :as xml]
            [camel-snake-kebab.core :as csk]))

(xml/alias-uri 'xh "http://www.w3.org/2001/XMLSchema")

(defn find-first [coll pred]
      (first (filter pred coll)))

(defn type->xsd-type [type]
      (cond
        (= type "instant") "a:date"
        :else (str "a:" (name type))))

(defmulti component->xsd-type :type)

(defmethod component->xsd-type :complex-type [{:keys [name children]}]
           [::xh/element {:name (csk/->PascalCase name)}
            [::xh/complexType
             [::xh/sequence
              (map component->xsd-type children)]]])

(defmethod component->xsd-type :simple-type [{:keys [name children]}]
           [::xh/element {:name (csk/->PascalCase name)}
            [::xh/simpleType
             [::xh/restriction {:base "a:string"}
              (map component->xsd-type children)]]])

(defmethod component->xsd-type :enumeration [{:keys [value]}]
           [::xh/enumeration {:value value}])

(defmethod component->xsd-type "uuid" [_]
           [::xh/element {:name "Uuid"}
            [::xh/simpleType
             [::xh/restriction {:base "a:string"}
              [::xh/length {:value 36
                            :fixed "true"}]
              [::xh/pattern {:value "[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-4[0-9a-fA-F]{3}-[8-9a-bA-B][0-9a-fA-F]{3}-[0-9a-fA-F]{12}"}]]]])

(defmethod component->xsd-type :default [{:keys [name type qualifiers]}]
           [::xh/element (cond-> {:name (csk/->PascalCase name)
                                  :type (type->xsd-type type)}

                                 (not-any? #(contains? qualifiers %) ['identity 'non-blank])
                                 (assoc :minOccurs 0)
                                 )])

(defn filter-query [model query start-c name]
      (let [component (second (find-first model #(= (first %) start-c)))
            c-name (or name (:horza.enum/name component) (:horza.entity/name component))
            type (:horza.model/type component)
            attrs (:horza.entity/attrs component)]
           (cond->
             {:name c-name
              :type (if (= type :horza.model/entity) :complex-type :simple-type)}
             (= type :horza.model/enum)
             (assoc :children
                    (map (fn [val] {:type  :enumeration
                                    :value (clojure.core/name (:horza.enum.type/value val))}) (:horza.enum/values component)))

             (not-empty query)
             (assoc :children
                    (map (fn [[h & t]]
                             (let [{:horza.attr/keys [name type relation qualifiers] :as attr} (find-first attrs #(= h (:horza.attr/name %)))]
                                  (if-not (nil? attr)
                                          (if-not (= type :horza.attr.type/ref)
                                                  {:name       (clojure.core/name name)
                                                   :type       (clojure.core/name type)
                                                   :qualifiers qualifiers}
                                                  (filter-query model t (second relation) (clojure.core/name name)))
                                          (throw (Exception. (str h " not found in model.")))))) query)))))

(defn model->xsd [model query start-component start-name]
      (-> [::xh/schema]
          (conj (component->xsd-type (filter-query model query start-component start-name)))
          xml/sexp-as-element))
