(ns com.webcomrades.horza.draw
  (:require [clojure.string :as str]
            [hiccup.core :as html]))

(defn escape
  [s]
  (str/replace s "-" "_"))

(defn attr->port
  [attr direction]
  (str (escape (namespace attr)) "____" (escape (name attr)) "_" (name direction)))

(defn entity->html
  [name doc attrs]
  (html/html [:table {:border 0 :cellborder 1 :cellspacing 0}
              [:tr [:td {:colspan 3} [:b name]]]
              [:tr [:td {:colspan 3} (when doc [:i doc])]]
              (for [{:horza.attr/keys [name type qualifiers] :as attr} (sort-by (juxt #(not (contains? (:horza.attr/qualifiers %) 'identity)) :horza.attr/name) attrs)]
                [:tr [:td {:align "left"} (clojure.core/name name)] [:td {:align "left"} (str (clojure.core/name type))] [:td {:align "left" :port (attr->port name :out)} (str/join "," qualifiers)]])]))

(defn graph-edge
  [from attr to]
  (str from ":" (attr->port attr :out) ":e -> " to "\n"))

(defn relationship->dot
  [from {:horza.attr/keys [name relation]}]
  (if (= :horza.ref/union (first relation))
    (apply str (map #(graph-edge from name %) (second relation)))
    (graph-edge from name (second relation))))

(defn entity-relations->dot
  [{:horza.entity/keys [name attrs]}]
  (into []
        (comp (filter #(= :horza.attr.type/ref (:horza.attr/type %)))
              (map (partial relationship->dot name)))
        attrs))

(defn enum->html
  [name values]
  (html/html [:table {:border 0 :cellborder 1 :cellspacing 0}
              [:tr [:td [:b name]]]
              (for [{:keys [horza.enum.type/value]} values]
                [:tr [:td {:align "left"} (clojure.core/name value)]])]))

(defmulti component->dot :horza.model/type)

(defmethod component->dot :horza.model/entity
  [{:horza.entity/keys [name doc attrs]}]
  (str name "[ " "label = <" (entity->html name doc attrs) "> ]\n"))

(defmethod component->dot :horza.model/enum
  [{:horza.enum/keys [name values]}]
  (str name "[ label = <" (enum->html name values) "> ]\n"))

(defn model->dot
  "Generate dotfile for graphviz"
  [model]
  (apply str
         "digraph G {\n"
         "\tfontname = \"Bitstream Vera Sans\"\n"
         "\tfontsize = 8\n\n"
         "\tnode [\n"
         "\t\tfontname = \"Bitstream Vera Sans\"\n"
         "\t\tfontsize = 8\n"
         "\t\tshape = \"plain\"\n"
         "\t]\n\n"
         "\tedge [\n"
         "\t\tfontname = \"Bitstream Vera Sans\"\n"
         "\t\tfontsize = 8\n"
         "\t]\n"
         (-> []
             (into (comp (map second) (map component->dot)) model)
             (into (comp (map second)
                         (filter #(= :horza.model/entity (:horza.model/type %)))
                         (mapcat entity-relations->dot)) model)
             (conj "}"))))

