(ns com.webcomrades.horza.core
  (:require [clojure.spec.alpha :as s]
            [medley.core :refer [map-vals]]
            [com.webcomrades.util.victorinox :refer [update-some]]
            [cognitect.anomalies :as anomalies]))

(s/def ::attribute
  (s/cat :horza.model/type #{'attr}
         :horza.attr/name keyword?
         :horza.attr/type (s/or :horza.attr.type/scalar #{'string 'instant 'uuid 'keyword 'long 'float 'double 'boolean 'bigint 'bigdec}
                                :horza.attr.type/ref (s/or :horza.ref/one symbol? :horza.ref/union (s/coll-of symbol? :kind set?)))
         :horza.attr/qualifiers (s/* #{'identity 'non-blank 'unique})
         :horza.attr/cardinality (s/? #{'many})
         :horza.attr/component (s/? #{'component})
         :horza.attr/doc (s/? string?)))

(s/def ::entity
  (s/cat :horza.model/type #{'entity}
         :horza.entity/name symbol?
         :horza.entity/doc (s/? string?)
         :horza.entity/attrs (s/* (s/spec ::attribute))))

(s/def ::enum-type
  (s/or :horza.enum/type (s/cat :horza.enum.type/value keyword? :horza.enum.type/doc (s/? string?))
        :horza.enum/type keyword?))

(s/def ::enum
  (s/cat :horza.model/type #{'enum}
         :horza.enum/name symbol?
         :horza.enum/doc (s/? string?)
         :horza.enum/values (s/coll-of ::enum-type :kind set?)))

(s/def ::model
  (s/coll-of (s/alt :horza.model/entity ::entity :horza.model/enum ::enum)))

(defn index-model-type
  [model-component]
  (-> model-component
      (update-some :horza.enum/values (fn [attrs]
                                        (map (fn [[_ v]] (if (map? v)
                                                           v
                                                           {:horza.enum.type/value v})) attrs)))
      (update-some :horza.entity/attrs (fn [attrs]
                                         (map (fn [{:horza.attr/keys [type] :as attr}]
                                                (-> (if (= (first type) :horza.attr.type/scalar)
                                                      (assoc attr :horza.attr/type (keyword "horza.attr.type.scalar" (name (second type))))
                                                      (assoc attr :horza.attr/type :horza.attr.type/ref
                                                                  :horza.attr/relation (second type)))

                                                    (dissoc :horza.model/type)
                                                    (update :horza.attr/qualifiers (partial into #{}))))
                                              attrs)))))

(defn index-model
  [model]
  (reduce (fn [model [component-type component]]
            (let [comp-name (get component (keyword (str "horza." (name component-type)) "name"))]
              (if (get model comp-name)
                (throw (ex-info (str "Model component with name " comp-name " already exists") {:model model :component component}))
                (assoc model
                  comp-name
                  (index-model-type (assoc component :horza.model/type component-type))))))
          {}
          model))

(defn verify-relationships
  [model]
  (let [errors (into [] (comp (map second)
                              (mapcat :horza.entity/attrs)
                              (filter #(= :horza.attr.type/ref (:horza.attr/type %)))
                              (mapcat (fn [{:horza.attr/keys [name relation]}]
                                        (let [[relation-type entity] relation]
                                          (if (= :horza.ref/union relation-type)
                                            (map (fn [e] [name e]) entity)
                                            [[name entity]]))))
                              (filter (fn [[attr-name ref-entity]]
                                        (not (get model ref-entity))))
                              (map (fn [[attr ref-entity]]
                                     {::anomalies/message  (str "Relation defined to a non-existent entity " ref-entity " on attribute " attr)
                                      ::anomalies/category ::anomalies/incorrect
                                      ::attr               attr
                                      ::entity             ref-entity})))
                     model)]
    (if (seq errors)
      (throw (ex-info "Invalid model" {::errors errors}))
      model)))

(defn parse
  "Parses a domain model"
  [model]
  (-> (s/conform ::model model)
      (index-model)
      (verify-relationships)))



