(ns com.walmartlabs.nexus-crate
  "Provides tools to download artifacts from artifact
  repositories. Currently, there are two supported methods:

  `get-maven-url` - get the url for a file on a maven
  repository (defaults to central)

  `get-nexus-url` - get the url for a file on a nexus repository. Uses
  the REST API therefore can find latest version of a SNAPSHOT"
  (:require [clojure.string :as string]
            [clojure.set :refer [rename-keys]]
            [pallet.api :as api]
            [pallet.actions :as actions]
            [pallet.crate :as crate]
            [pallet.environment :as env]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ## Helpers

(defn- select-rename-keys
  "Returns a map containing only those entries in map whose key is in
  keyseq. A key may optionally be a vector of two keywords
  [key renamed-key]

  (select-rename-keys {:a 1 :b 2 :c 3} [:a [:c :d]]) => {:a 1 :d 3}"
  [m keyseq]
  (let [ks (map (fn [x]
                  (if (vector? x) (first x) x))
                keyseq)
        renames (apply hash-map (flatten (filter vector? keyseq)))]
    (-> (select-keys m ks)
        (rename-keys renames))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ## Settings

(defn- merge-settings [settings]
  (merge (env/get-environment [:nexus])
         settings))

(defn- check-settings
  [{:keys [repositories default-repository]}]
  (assert (every? (fn [[repository-id {:keys [endpoint repository]}]]
                    (and (keyword? repository-id)
                         (string? endpoint)
                         (string? repository)))
                  repositories)
          "Repositories should be a map of keyword (repo id) to repository spec")
  (assert (or (and (= (count repositories) 1))
              (and (> (count repositories) 1)
                   default-repository))
          "If more than one repository is specified, default-repository must be present")
  (when default-repository
    (assert (contains? repositories default-repository))))

(defn- ensure-default-repository
  [{:keys [repositories default-repository] :as settings}]
  (assoc settings
    :default-repository (or default-repository (ffirst repositories))))

(crate/defplan settings
  [options]
  (let [settings (merge-settings options)]
    (when (seq settings)
      (check-settings settings)
      (actions/assoc-settings :nexus (ensure-default-repository settings)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ## Nexus URL

(def ^:private rest-path
  "/service/local/artifact/maven/redirect")

(defn- destruct-dep
  "Converts a dependency definition from a shortcut lein style into a
  full map"
  [[id version & options]]
  {:pre [(symbol? id) (string? version)]}
  (-> (apply hash-map options)
      (assoc :group-id (or (namespace id)
                           (name id))
             :artifact-id (name id)
             :version version)))

(defn- repository-map
  [settings repository-id]
  (let [{:keys [repositories default-repository]} settings
        repository-id (or repository-id default-repository)]
    (get repositories repository-id)))

(defn- params-str
  [params-map]
  (->> params-map
       (map (fn [[k v]]
              (str (name k) \= v)))
       (string/join "&")))

(defn params-map
  [artifact repository]
  (-> artifact
      (select-rename-keys [[:group-id :g]
                           [:artifact-id :a]
                           [:version :v]
                           [:classifier :c]
                           [:extension :e]
                           [:packaging :p]])
      (assoc :r repository)))

(defn- make-nexus-url
  "Takes an artifact definition in lein style and converts it to the
  url where the artifact can be found on the artifact repository"
  [settings artifact]
  (let [repository-id (:repository-id artifact)
        repository-map (repository-map settings repository-id)
        {:keys [endpoint repository]} repository-map
        params (params-str (params-map artifact repository))]
    (assert repository-map)
    (assert params)
    (format "%s%s?%s" endpoint rest-path params)))

(defn get-nexus-url
  "Returns the url that the artifact can be retrieved from. Uses Rest
  API as opposed to directly looking up a url. Needed for example for
  looking up latest version of a snapshot. Useful when used as part of
  `(pallet.actions/remote-file :url (get-nexus-url artifact))`

  `artifact` - ['group/artifact version & settings] (leiningen format)
    settings:
     :classifier - optional (see \"http://maven.apache.org/pom.html\")
     :repository - The url for the repository to to download the
                   artifact from
     :repository-id - The repository to download the artifact
                      from. Should be a keyword that is contained in
                      the repository settings for this crate. If not
                      included, the :default-repository will be
                      used. See `server-spec` in this ns for settings
                      info

  `settings` - Optional map of settings. See server-spec doc"
  ([artifact]
     (get-nexus-url artifact (crate/get-settings :nexus)))
  ([artifact settings]
     {:pre [artifact
            (vector? artifact)
            (symbol? (first artifact))
            (string? (second artifact))]}
     (check-settings settings)
     (make-nexus-url settings (destruct-dep artifact))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Maven URL

(defn- artifact-path
  [artifact]
  (let [{:keys [artifact-id group-id version extension]} artifact
        group-dir (string/replace group-id \. \/)]
    (format "/%s/%s/%s/%s-%s.%s"
            group-dir
            artifact-id
            version
            artifact-id
            version
            extension)))

(defn get-maven-url
  "Returns the url of the artifact on a maven repository. Useful when
  used as part of
  `(pallet.actions/remote-file :url (get-maven-url artifact))`

  `artifact` - ['group/artifact version & settings] (leiningen format)
    settings:
     :extension  - optional. Defaults to jar

  `repository-url` - The repository to download the artifact from"
  [artifact & {:keys [repository-url]
               :or {repository-url "http://central.maven.org/maven2"}}]
  {:pre [artifact
         (vector? artifact)
         (symbol? (first artifact))
         (string? (second artifact))
         (string? repository-url)]}
  (let [default-artifact-fields {:extension "jar"}
        artifact-map (merge default-artifact-fields
                            (destruct-dep artifact))]
    (str repository-url
         (artifact-path artifact-map))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ## Spec

(defn server-spec
  "Options can either be in the environment map under :nexus, or they
  can be passed in here

  `:repositories` - a map of repository id (keyword) to repository. At
  least one entry required
    repository:
     :endpoint - E.g \"http://foo.bar:8010\"
     :repository - E.g \"snapshots\"

  `:default-remote-repository` - Optional if there is only one entry
  in :repositories. Otherwise, a keyword that must be contained
  in :repositories"
  [& {:keys [remote-repositories default-repository] :as options}]
  (api/server-spec
   :phases {:settings (api/plan-fn (settings options))}))
