(ns leiningen.sealog.impl
  (:require [clojure.edn :as edn]
            [clojure.spec.alpha :as spec]
            [clojure.string :as str]
            [com.wallbrew.spoon.core :as spoon]
            [leiningen.core.main :as main]
            [leiningen.sealog.impl.io :as io]
            [leiningen.sealog.types.changelog :as changelog]
            [leiningen.sealog.types.config :as config]
            [spec-tools.core :as st]))


(defn select-config
  "Select the configuration to use with the following precedence:
    - The `:sealog` key in project.clj
    - The configuration file in .sealog/config.edn
    - The configuration file in .wallbrew/sealog/config.edn
    - The default configuration"
  [project]
  (let [project-config             (:sealog project)
        config-file-exists?        (io/file-exists? config/config-file)
        backup-config-file-exists? (io/file-exists? config/backup-config-file)]
    (cond
      (map? project-config)      project-config
      config-file-exists?        (io/read-edn-file! config/config-file ::config/config)
      backup-config-file-exists? (io/read-edn-file! config/backup-config-file ::config/config)
      :else                      (do (main/info "No configuration file found. Assuming default configuration.")
                                     config/default-config))))


(defn load-config!
  "Load the configuration file with the following precedence:
    - The `:sealog` key in project.clj
    - The configuration file in .sealog/config.edn
    - The default configuration

   If the configuration is invalid, print a warning and exit."
  [project]
  (let [config (select-config project)]
    (if (spec/valid? ::config/config config)
      config
      (do (main/warn (format "Invalid configuration file contents: %s" (spec/explain-str ::config/config config)))
          (main/exit 1)))))


(defn load-changelog-entry-directory!
  "Load the changelog directory into a map of version to changelog entries."
  [{:keys [changelog-entry-directory] :as _config}]
  (let [files   (io/list-all-files changelog-entry-directory)
        reducer (fn [acc file]
                  (let [content (io/read-edn-file! file ::changelog/entry)]
                    (conj acc content)))]
    (reduce reducer [] files)))


(defn sealog-initialized?
  "Returns true if the sealog directory exists."
  [{:keys [changelog-entry-directory] :as config}]
  (if (io/file-exists? changelog-entry-directory)
    (boolean (seq (load-changelog-entry-directory! config)))
    false))


(defn render-preamble
  "Render the preamble of the changelog."
  []
  ["# Changelog"
   ""
   "All notable changes to this project will be documented in this file."
   ""
   "The format is based on [Keep a Changelog](https://keepachangelog.com/en/1.0.0/)"
   ""])


(defn render-footer
  "Render the footer of the changelog, and a notice about changelog tooling."
  []
  ["## Source"
   ""
   "This changelog was generated by [sealog.](https://github.com/Wall-Brew-Co/lein-sealog)"
   "Please do not edit it directly. Instead, edit the source data files and regenerate this file."
   ""])


(defn ->entry-title
  "Convert a changelog entry into a title for the changelog."
  [{:keys [timestamp] :as change}]
  (let [friendly-time    (first (str/split (str timestamp) #"T"))
        friendly-version (changelog/render-version change)]
    (format "%s - %s" friendly-version friendly-time)))


(defn render-toc
  "Render the table of contents for the changelog."
  [changelog]
  (let [->header (fn [entry]
                   (let [entry-title (->entry-title entry)
                         entry-link  (str/replace (str/replace entry-title #" " "-") #"\." "")]
                     (format "* [%s](#%s)" entry-title entry-link)))
        toc      (mapv ->header changelog)]
    (spoon/concatv ["## Table of Contents" ""] toc [""])))


(defn render-change-set
  "Render the set of changes for a given version."
  [{:keys [added changed deprecated removed fixed security]}]
  (let [->change-entry (fn [change] (format "  * %s" change))
        ->change       (fn [change-type changes]
                         (if (seq changes)
                           (let [heading (format "* %s" change-type)
                                 entries (mapv ->change-entry changes)]
                             (spoon/concatv [heading] entries))
                           []))]
    (spoon/concatv (->change "Added" added)
                   (->change "Changed" changed)
                   (->change "Deprecated" deprecated)
                   (->change "Removed" removed)
                   (->change "Fixed" fixed)
                   (->change "Security" security))))


(defn render-changelog-entry
  "Render the changelog entry for a specific version."
  [{:keys [changes] :as change}]
  (let [header      (str "## " (->entry-title change))
        change-list (render-change-set changes)]
    (spoon/concatv [header ""] change-list [""])))


(defn render-changes
  "Render the changes for a given changelog."
  [changelog]
  (flatten (mapv render-changelog-entry changelog)))


(defn render-changelog
  "Render the changelog."
  [changelog]
  (let [preamble (render-preamble)
        toc      (render-toc changelog)
        changes  (render-changes changelog)
        footer   (render-footer)]
    (str/join "\n" (spoon/concatv preamble toc changes footer))))


(defn init!
  "Create a new changelog directory."
  [{:keys [changelog-entry-directory version-scheme] :as config}]
  (let [initial-entry          (changelog/initialize version-scheme)
        initial-entry-filename (str changelog-entry-directory (changelog/render-filename initial-entry))
        entry                  (update initial-entry :timestamp str)]
    (io/create-file initial-entry-filename)
    (io/write-edn-file! initial-entry-filename entry config)))


(defn sealog-configured?
  "Returns true if the sealog configuration exists either in project.clj or in the configuration file."
  [project]
  (or (:sealog project)
      (io/file-exists? config/config-file)
      (io/file-exists? config/backup-config-file)))


(defn configure!
  "Create a new configuration file."
  [_opts]
  (io/create-file config/config-file)
  (io/write-file! config/config-file config/default-config))


(defn valid-configuration?
  "Returns true if the configuration is valid."
  [project]
  (let [configuration-contents (select-config project)
        contents               (st/coerce ::config/config configuration-contents st/string-transformer)
        valid?                 (spec/valid? ::config/config contents)]
    (if valid?
      (do (main/info "Sealog configuration is valid.")
          true)
      (do (main/warn (format "Invalid configuration file contents: %s" (spec/explain-str ::config/config contents)))
          false))))


(defn changelog-entry-directory-is-not-empty?
  "Returns true if the changelog entry directory is not empty."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [has-files? (boolean (seq (io/list-all-files changelog-entry-directory)))]
    (if has-files?
      (do (main/info "Changelog entry directory contains at least one file.")
          true)
      (do (main/warn "Changelog entry directory is empty.")
          false))))


(defn changelog-directory-only-contains-valid-files?
  "Returns true if the changelog entry directory only contains valid files."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [files      (io/list-all-files changelog-entry-directory)
        valid?     (fn [filepath]
                     (let [file-content (edn/read-string (io/read-file! filepath))
                           contents     (st/coerce ::changelog/entry file-content st/string-transformer)]
                       (if (spec/valid? ::changelog/entry contents)
                         true
                         (do (main/warn (format "Invalid changelog file contents at path `%s`: %s"
                                                filepath
                                                (spec/explain-str ::changelog/entry contents)))
                             false))))
        all-valid? (every? valid? files)]
    (if all-valid?
      (do (main/info "All changelog entries are valid.")
          true)
      false)))


(defn all-changelog-entries-use-same-version-type?
  "Returns true if all changelog entries use the same version type."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [files         (io/list-all-files changelog-entry-directory)
        reducer       (fn [acc filepath]
                        (let [content (edn/read-string (io/read-file! filepath))]
                          (conj acc (:version-type content))))
        version-types (vec (distinct (reduce reducer [] files)))]
    (if (= 1 (count version-types))
      (do (main/info "All changelog entries use the same version type.")
          true)
      (do (main/warn (format "Changelog entries use multiple version types: %s" version-types))
          false))))


(defn all-changelog-entries-have-distinct-versions?
  "Returns true if all changelog entries have distinct versions."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [files         (io/list-all-files changelog-entry-directory)
        reducer       (fn [acc filepath]
                        (let [content (edn/read-string (io/read-file! filepath))]
                          (conj acc (:version content))))
        versions      (vec (distinct (reduce reducer [] files)))]
    (if (= (count files) (count versions))
      (do (main/info "All changelog entries have distinct versions.")
          true)
      (do (main/warn (format "Changelog entries have non-distinct versions: %s" versions))
          false))))


(defn project-version-matches-latest-changelog-entry?
  "Returns true if the project version matches the latest changelog entry."
  [project configuration]
  (let [changelog              (load-changelog-entry-directory! configuration)
        latest-changelog-entry (changelog/max-version changelog)
        sealog-version         (changelog/render-version latest-changelog-entry)
        leiningen-version      (:version project)]
    (if (= leiningen-version sealog-version)
      (do (main/info "Project version matches latest changelog entry.")
          true)
      (do (main/warn (format "Project version `%s` does not match latest changelog entry `%s`"
                             leiningen-version
                             sealog-version))
          false))))


(defn rendered-changelog-contains-all-changelog-entries?
  "Returns true if the rendered changelog contains all changelog entries."
  [{:keys [changelog-filename] :as configuration}]
  (let [changelog                   (load-changelog-entry-directory! configuration)
        rendered-changelog          (render-changelog changelog)
        rendered-changelog-contents (slurp changelog-filename)]
    (if (= (count rendered-changelog) (count rendered-changelog-contents))
      (do (main/info "Rendered changelog contains all changelog entries.")
          true)
      (do (main/warn "Rendered changelog does not contain all changelog entries. Please run `lein sealog render`.")
          false))))
