(ns leiningen.sealog.impl
  (:require [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.pprint :as pp]
            [clojure.spec.alpha :as spec]
            [clojure.string :as str]
            [com.wallbrew.spoon.core :as spoon]
            [leiningen.core.main :as main]
            [leiningen.sealog.types.changelog :as changelog]
            [leiningen.sealog.types.config :as config]
            [spec-tools.core :as st]))


(defn list-all-files
  "Recursively list all files within a directory."
  [path]
  (let [files  (io/file path)
        dir?   (fn [f] (.isDirectory f))
        ->path (fn [f] (.getPath f))]
    (if (dir? files)
      (mapv ->path (filter (complement dir?) (file-seq files)))
      (throw (ex-info "Not a directory!" {:path path})))))


(defn file-exists?
  "Returns true if `path` points to a valid file"
  [path]
  (and (string? path)
       (.exists (io/file path))))


(defn write-file!
  "This is a wrapper around `spit` that logs the filename to the console."
  [filename content]
  (main/info (format "Writing to %s" filename))
  (spit filename content))


(defn write-edn-file!
  "Write the contents to a file as EDN."
  [filename content {:keys [pretty-print-edn?]}]
  (if pretty-print-edn?
    (write-file! filename (with-out-str (pp/pprint content)))
    (write-file! filename content)))


(defn read-file!
  "This is a wrapper around `slurp` that logs the filename to the console."
  [filename]
  (main/info (format "Reading from %s" filename))
  (slurp filename))


(defn read-edn-file!
  "Reads an EDN file and returns the contents as a map.
   Throws an exception if the file does not exist, or if the contents do not coerce"
  [filename spec]
  (if (file-exists? filename)
    (let [file-content (edn/read-string (read-file! filename))
          contents     (st/coerce spec file-content st/string-transformer)]
      (if (spec/valid? spec contents)
        contents
        (throw (ex-info (str "Invalid file contents: " filename)
                        {:filename filename
                         :errors   (spec/explain-data spec contents)}))))
    (throw (ex-info "Not matching file exists!"
                    {:filename filename}))))


(defn load-config!
  "Load the configuration file."
  []
  (if (file-exists? config/config-file)
    (read-edn-file! config/config-file ::config/config)
    (do (main/info "No configuration file found. Assuming default configuration.")
        config/default-config)))


(defn load-changelog-entry-directory!
  "Load the changelog directory into a map of version to changelog entries."
  [{:keys [changelog-entry-directory] :as _config}]
  (let [files   (list-all-files changelog-entry-directory)
        reducer (fn [acc file]
                  (let [content (read-edn-file! file ::changelog/entry)]
                    (conj acc content)))]
    (reduce reducer [] files)))


(defn sealog-initialized?
  "Returns true if the sealog directory exists."
  [{:keys [changelog-entry-directory] :as config}]
  (if (file-exists? changelog-entry-directory)
    (boolean (seq (load-changelog-entry-directory! config)))
    false))


(defn render-preamble
  "Render the preamble of the changelog."
  []
  ["# Changelog"
   ""
   "All notable changes to this project will be documented in this file."
   ""
   "The format is based on [Keep a Changelog](https://keepachangelog.com/en/1.0.0/)"
   ""])


(defn render-footer
  "Render the footer of the changelog, and a notice about changelog tooling."
  []
  ["## Source"
   ""
   "This changelog was generated by [sealog.](https://github.com/Wall-Brew-Co/lein-sealog)"
   "Please do not edit it directly. Instead, edit the source data files and regenerate this file."
   ""])


(defn ->entry-title
  "Convert a changelog entry into a title for the changelog."
  [{:keys [timestamp] :as change}]
  (let [friendly-time    (first (str/split (str timestamp) #"T"))
        friendly-version (changelog/render-version change)]
    (format "%s - %s" friendly-version friendly-time)))


(defn render-toc
  "Render the table of contents for the changelog."
  [changelog]
  (let [->header (fn [entry]
                   (let [entry-title (->entry-title entry)
                         entry-link  (str/replace (str/replace entry-title #" " "-") #"\." "")]
                     (format "* [%s](#%s)" entry-title entry-link)))
        toc      (mapv ->header changelog)]
    (spoon/concatv ["## Table of Contents" ""] toc [""])))


(defn render-change-set
  "Render the set of changes for a given version."
  [{:keys [added changed deprecated removed fixed security]}]
  (let [->change-entry (fn [change] (format "  * %s" change))
        ->change       (fn [change-type changes]
                         (if (seq changes)
                           (let [heading (format "* %s" change-type)
                                 entries (mapv ->change-entry changes)]
                             (spoon/concatv [heading] entries))
                           []))]
    (spoon/concatv (->change "Added" added)
                   (->change "Changed" changed)
                   (->change "Deprecated" deprecated)
                   (->change "Removed" removed)
                   (->change "Fixed" fixed)
                   (->change "Security" security))))


(defn render-changelog-entry
  "Render the changelog entry for a specific version."
  [{:keys [changes] :as change}]
  (let [header      (str "## " (->entry-title change))
        change-list (render-change-set changes)]
    (spoon/concatv [header ""] change-list [""])))


(defn render-changes
  "Render the changes for a given changelog."
  [changelog]
  (flatten (mapv render-changelog-entry changelog)))


(defn render-changelog
  "Render the changelog."
  [changelog]
  (let [preamble (render-preamble)
        toc      (render-toc changelog)
        changes  (render-changes changelog)
        footer   (render-footer)]
    (str/join "\n" (spoon/concatv preamble toc changes footer))))


(defn init!
  "Create a new changelog directory."
  [{:keys [changelog-entry-directory version-scheme] :as config}]
  (let [initial-entry          (changelog/initialize version-scheme)
        initial-entry-filename (str changelog-entry-directory (changelog/render-filename initial-entry))
        entry                  (update initial-entry :timestamp str)]
    (io/make-parents (io/file initial-entry-filename))
    (write-edn-file! initial-entry-filename entry config)))


(defn sealog-configured?
  "Returns true if the sealog configuration file exists."
  []
  (file-exists? config/config-file))


(defn configure!
  "Create a new configuration file."
  [_opts]
  (io/make-parents (io/file config/config-file))
  (write-file! config/config-file config/default-config))


(defn valid-configuration?
  "Returns true if the configuration is valid."
  []
  (let [configuration-contents (if (file-exists? config/config-file)
                                 (edn/read-string (read-file! config/config-file))
                                 config/default-config)
        contents               (st/coerce ::config/config configuration-contents st/string-transformer)
        valid?                 (spec/valid? ::config/config contents)]
    (if valid?
      (do (main/info "Sealog configuration is valid.")
          true)
      (do (main/warn (format "Invalid configuration file contents: %s" (spec/explain-str ::config/config contents)))
          false))))


(defn changelog-entry-directory-is-not-empty?
  "Returns true if the changelog entry directory is not empty."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [has-files? (boolean (seq (list-all-files changelog-entry-directory)))]
    (if has-files?
      (do (main/info "Changelog entry directory contains at least one file.")
          true)
      (do (main/warn "Changelog entry directory is empty.")
          false))))


(defn changelog-directory-only-contains-valid-files?
  "Returns true if the changelog entry directory only contains valid files."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [files      (list-all-files changelog-entry-directory)
        valid?     (fn [filepath]
                     (let [file-content (edn/read-string (read-file! filepath))
                           contents     (st/coerce ::changelog/entry file-content st/string-transformer)]
                       (if (spec/valid? ::changelog/entry contents)
                         true
                         (do (main/warn (format "Invalid changelog file contents at path `%s`: %s"
                                                filepath
                                                (spec/explain-str ::changelog/entry contents)))
                             false))))
        all-valid? (every? valid? files)]
    (if all-valid?
      (do (main/info "All changelog entries are valid.")
          true)
      false)))


(defn all-changelog-entries-use-same-version-type?
  "Returns true if all changelog entries use the same version type."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [files         (list-all-files changelog-entry-directory)
        reducer       (fn [acc filepath]
                        (let [content (edn/read-string (read-file! filepath))]
                          (conj acc (:version-type content))))
        version-types (vec (distinct (reduce reducer [] files)))]
    (if (= 1 (count version-types))
      (do (main/info "All changelog entries use the same version type.")
          true)
      (do (main/warn (format "Changelog entries use multiple version types: %s" version-types))
          false))))


(defn all-changelog-entries-have-distinct-versions?
  "Returns true if all changelog entries have distinct versions."
  [{:keys [changelog-entry-directory] :as _configuration}]
  (let [files         (list-all-files changelog-entry-directory)
        reducer       (fn [acc filepath]
                        (let [content (edn/read-string (read-file! filepath))]
                          (conj acc (:version content))))
        versions      (vec (distinct (reduce reducer [] files)))]
    (if (= (count files) (count versions))
      (do (main/info "All changelog entries have distinct versions.")
          true)
      (do (main/warn (format "Changelog entries have non-distinct versions: %s" versions))
          false))))


(defn project-version-matches-latest-changelog-entry?
  "Returns true if the project version matches the latest changelog entry."
  [project configuration]
  (let [changelog              (load-changelog-entry-directory! configuration)
        latest-changelog-entry (changelog/max-version changelog)
        sealog-version         (changelog/render-version latest-changelog-entry)
        leiningen-version      (:version project)]
    (if (= leiningen-version sealog-version)
      (do (main/info "Project version matches latest changelog entry.")
          true)
      (do (main/warn (format "Project version `%s` does not match latest changelog entry `%s`"
                             leiningen-version
                             sealog-version))
          false))))


(defn rendered-changelog-contains-all-changelog-entries?
  "Returns true if the rendered changelog contains all changelog entries."
  [{:keys [changelog-filename] :as configuration}]
  (let [changelog                   (load-changelog-entry-directory! configuration)
        rendered-changelog          (render-changelog changelog)
        rendered-changelog-contents (slurp changelog-filename)]
    (if (= (count rendered-changelog) (count rendered-changelog-contents))
      (do (main/info "Rendered changelog contains all changelog entries.")
          true)
      (do (main/warn "Rendered changelog does not contain all changelog entries. Please run `lein sealog render`.")
          false))))
