(ns leiningen.sealog.impl
  (:require [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [leiningen.sealog.config :as config]
            [leiningen.sealog.types.changelog :as changelog]
            [spec-tools.core :as st]))


(defn concatv
  "Concatenates the given sequences into a vector.
   This is a workaround for the fact that `concat` returns a lazy sequence."
  [& vectors]
  (vec (apply concat vectors)))


(defn list-all-files
  "Recursively list all files within a directory."
  [path]
  (let [files  (io/file path)
        dir?   (fn [f] (.isDirectory f))
        ->path (fn [f] (.getPath f))]
    (if (dir? files)
      (mapv ->path (filter (complement dir?) (file-seq files)))
      (throw (ex-info "Not a directory!" {:path path})))))


(defn file-exists?
  "Returns true if `path` points to a valid file"
  [path]
  (and (string? path)
       (.exists (io/file path))))


(defn read-edn-file!
  "Reads an EDN file and returns the contents as a map.
   Throws an exception if the file does not exist, or if the contents do not coerce"
  [filename spec]
  (if (file-exists? filename)
    (let [contents (st/coerce spec (edn/read-string (slurp filename)) st/string-transformer)]
      (if (s/valid? spec contents)
        contents
        (throw (ex-info (str "Invalid file contents: " filename)
                        {:filename filename
                         :errors   (s/explain-data spec contents)}))))
    (throw (ex-info "Not matching file exists!"
                    {:filename filename}))))


(defn load-changelog-directory!
  "Load the changelog directory into a map of version to changelog entries."
  []
  (let [files   (list-all-files config/changelog-directory)
        reducer (fn [acc file]
                  (let [content (read-edn-file! file ::changelog/entry)]
                    (conj acc content)))]
    (reduce reducer [] files)))


(defn sealog-initialized?
  "Returns true if the sealog directory exists."
  []
  (if (file-exists? config/changelog-directory)
    (boolean (seq (load-changelog-directory!)))
    false))


(defn render-preamble
  "Render the preamble of the changelog."
  []
  ["# Changelog"
   ""
   "All notable changes to this project will be documented in this file."
   ""
   "The format is based on [Keep a Changelog](https://keepachangelog.com/en/1.0.0/)"
   ""])


(defn render-footer
  "Render the footer of the changelog, and a notice about changelog tooling."
  []
  ["## Source"
   ""
   "This change log was generated by [sealog.](https://github.com/Wall-Brew-Co/lein-sealog)"
   "Please do not edit it directly. Instead, edit the source data files and regenerate this file."
   ""])


(defn ->entry-title
  "Convert a changelog entry into a title for the changelog."
  [{:keys [timestamp] :as change}]
  (let [friendly-time    (first (str/split (str timestamp) #"T"))
        friendly-version (changelog/render-version change)]
    (format "%s - %s" friendly-version friendly-time)))


(defn render-toc
  "Render the table of contents for the changelog."
  [changelog]
  (let [->header (fn [entry]
                   (let [entry-title (->entry-title entry)
                         entry-link  (str/replace (str/replace entry-title #" " "-") #"\." "")]
                     (format "* [%s](#%s)" entry-title entry-link)))
        toc      (mapv ->header changelog)]
    (concatv ["## Table of Contents" ""] toc [""])))


(defn render-change-set
  "Render the set of changes for a given version."
  [{:keys [added changed deprecated removed fixed security]}]
  (let [->change-entry (fn [change] (format "  * %s" change))
        ->change       (fn [change-type changes]
                         (if (seq changes)
                           (let [heading (format "* %s" change-type)
                                 entries (mapv ->change-entry changes)]
                             (concatv [heading] entries))
                           []))]
    (concatv (->change "Added" added)
             (->change "Changed" changed)
             (->change "Deprecated" deprecated)
             (->change "Removed" removed)
             (->change "Fixed" fixed)
             (->change "Security" security))))


(defn render-changelog-entry
  "Render the changelog entry for a specific version."
  [{:keys [changes] :as change}]
  (let [header      (str "## " (->entry-title change))
        change-list (render-change-set changes)]
    (concatv [header ""] change-list [""])))


(defn render-changes
  "Render the changes for a given changelog."
  [changelog]
  (flatten (mapv render-changelog-entry changelog)))


(defn render-changelog
  "Render the changelog."
  [changelog]
  (let [preamble (render-preamble)
        toc      (render-toc changelog)
        changes  (render-changes changelog)
        footer   (render-footer)]
    (str/join "\n" (concatv preamble toc changes footer))))


(defn init!
  "Create a new changelog directory."
  []
  (let [initial-entry (changelog/initialize config/default-scheme)
        filename      (str config/changelog-directory (changelog/render-filename initial-entry))
        entry         (update initial-entry :timestamp str)]
    (io/make-parents (io/file filename))
    (spit filename entry)))
