// Copyright 2008 The Closure Library Authors. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS-IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/**
 * @fileoverview Renderer for {@link goog.ui.MenuButton}s and subclasses.
 *
 * @author attila@google.com (Attila Bodis)
 */

goog.provide('goog.ui.MenuButtonRenderer');

goog.require('goog.a11y.aria');
goog.require('goog.a11y.aria.State');
goog.require('goog.asserts');
goog.require('goog.dom');
goog.require('goog.string');
goog.require('goog.style');
goog.require('goog.ui.Component');
goog.require('goog.ui.CustomButtonRenderer');
goog.require('goog.ui.INLINE_BLOCK_CLASSNAME');
goog.require('goog.ui.Menu');
goog.require('goog.ui.MenuRenderer');
goog.require('goog.userAgent');



/**
 * Renderer for {@link goog.ui.MenuButton}s.  This implementation overrides
 * {@link goog.ui.CustomButtonRenderer#createButton} to create a separate
 * caption and dropdown element.
 * @constructor
 * @extends {goog.ui.CustomButtonRenderer}
 */
goog.ui.MenuButtonRenderer = function() {
  goog.ui.CustomButtonRenderer.call(this);
};
goog.inherits(goog.ui.MenuButtonRenderer, goog.ui.CustomButtonRenderer);
goog.addSingletonGetter(goog.ui.MenuButtonRenderer);


/**
 * Default CSS class to be applied to the root element of components rendered
 * by this renderer.
 * @type {string}
 */
goog.ui.MenuButtonRenderer.CSS_CLASS = goog.getCssName('goog-menu-button');


/**
 * A property to denote content elements that have been wrapped in an extra
 * div to work around FF2/RTL bugs.
 * @type {string}
 * @private
 */
goog.ui.MenuButtonRenderer.WRAPPER_PROP_ = '__goog_wrapper_div';


if (goog.userAgent.GECKO) {
  /**
   * Takes the menubutton's root element, and sets its content to the given
   * text caption or DOM structure. Because the DOM structure of this button is
   * conditional based on whether we need to work around FF2/RTL bugs, we
   * override the default implementation to take this into account.
   * @param {Element} element The control's root element.
   * @param {goog.ui.ControlContent} content Text caption or DOM
   *     structure to be set as the control's content.
   * @override
   */
  goog.ui.MenuButtonRenderer.prototype.setContent = function(element,
      content) {
    var caption =
        goog.ui.MenuButtonRenderer.superClass_.getContentElement.call(this,
            /** @type {Element} */ (element && element.firstChild));
    if (caption) {
      goog.dom.replaceNode(
          this.createCaption(content, goog.dom.getDomHelper(element)),
          caption);
    }
  };
} // end goog.userAgent.GECKO


/**
 * Takes the button's root element and returns the parent element of the
 * button's contents.  Overrides the superclass implementation by taking
 * the nested DIV structure of menu buttons into account.
 * @param {Element} element Root element of the button whose content element
 *     is to be returned.
 * @return {Element} The button's content element.
 * @override
 */
goog.ui.MenuButtonRenderer.prototype.getContentElement = function(element) {
  var content =
      goog.ui.MenuButtonRenderer.superClass_.getContentElement.call(this,
          /** @type {Element} */ (element && element.firstChild));
  if (goog.userAgent.GECKO && content &&
      content[goog.ui.MenuButtonRenderer.WRAPPER_PROP_]) {
    content = /** @type {Element} */ (content.firstChild);
  }
  return content;
};


/**
 * Updates the menu button's ARIA (accessibility) state so that aria-expanded
 * does not appear when the button is "opened."
 * @param {Element} element Element whose ARIA state is to be updated.
 * @param {goog.ui.Component.State} state Component state being enabled or
 *     disabled.
 * @param {boolean} enable Whether the state is being enabled or disabled.
 * @protected
 * @override
 */
goog.ui.MenuButtonRenderer.prototype.updateAriaState = function(element, state,
    enable) {
  // If button has OPENED state, do not assign an ARIA state. Usually
  // aria-expanded would be assigned, which does not mean anything for a menu
  // button.
  goog.asserts.assert(
      element, 'The menu button DOM element cannot be null.');
  goog.asserts.assert(
      goog.string.isEmpty(
      goog.a11y.aria.getState(element, goog.a11y.aria.State.EXPANDED)),
      'Menu buttons do not support the ARIA expanded attribute. ' +
      'Please use ARIA disabled instead.' +
      goog.a11y.aria.getState(element, goog.a11y.aria.State.EXPANDED).length);
  if (state != goog.ui.Component.State.OPENED) {
    goog.base(this, 'updateAriaState', element, state, enable);
  }
};


/**
 * Takes an element, decorates it with the menu button control, and returns
 * the element.  Overrides {@link goog.ui.CustomButtonRenderer#decorate} by
 * looking for a child element that can be decorated by a menu, and if it
 * finds one, decorates it and attaches it to the menu button.
 * @param {goog.ui.Control} control goog.ui.MenuButton to decorate the element.
 * @param {Element} element Element to decorate.
 * @return {Element} Decorated element.
 * @override
 */
goog.ui.MenuButtonRenderer.prototype.decorate = function(control, element) {
  var button = /** @type {goog.ui.MenuButton} */ (control);
  // TODO(attila):  Add more robust support for subclasses of goog.ui.Menu.
  var menuElem = goog.dom.getElementsByTagNameAndClass(
      '*', goog.ui.MenuRenderer.CSS_CLASS, element)[0];
  if (menuElem) {
    // Move the menu element directly under the body (but hide it first to
    // prevent flicker; see bug 1089244).
    goog.style.setElementShown(menuElem, false);
    goog.dom.appendChild(goog.dom.getOwnerDocument(menuElem).body, menuElem);

    // Decorate the menu and attach it to the button.
    var menu = new goog.ui.Menu();
    menu.decorate(menuElem);
    button.setMenu(menu);
  }

  // Let the superclass do the rest.
  return goog.ui.MenuButtonRenderer.superClass_.decorate.call(this, button,
      element);
};


/**
 * Takes a text caption or existing DOM structure, and returns the content and
 * a dropdown arrow element wrapped in a pseudo-rounded-corner box.  Creates
 * the following DOM structure:
 *    <div class="goog-inline-block goog-menu-button-outer-box">
 *      <div class="goog-inline-block goog-menu-button-inner-box">
 *        <div class="goog-inline-block goog-menu-button-caption">
 *          Contents...
 *        </div>
 *        <div class="goog-inline-block goog-menu-button-dropdown">
 *          &nbsp;
 *        </div>
 *      </div>
 *    </div>
 * @param {goog.ui.ControlContent} content Text caption or DOM structure
 *     to wrap in a box.
 * @param {goog.dom.DomHelper} dom DOM helper, used for document interaction.
 * @return {Element} Pseudo-rounded-corner box containing the content.
 * @override
 */
goog.ui.MenuButtonRenderer.prototype.createButton = function(content, dom) {
  return goog.ui.MenuButtonRenderer.superClass_.createButton.call(this,
      [this.createCaption(content, dom), this.createDropdown(dom)], dom);
};


/**
 * Takes a text caption or existing DOM structure, and returns it wrapped in
 * an appropriately-styled DIV.  Creates the following DOM structure:
 *    <div class="goog-inline-block goog-menu-button-caption">
 *      Contents...
 *    </div>
 * @param {goog.ui.ControlContent} content Text caption or DOM structure
 *     to wrap in a box.
 * @param {goog.dom.DomHelper} dom DOM helper, used for document interaction.
 * @return {Element} Caption element.
 */
goog.ui.MenuButtonRenderer.prototype.createCaption = function(content, dom) {
  return goog.ui.MenuButtonRenderer.wrapCaption(
      content, this.getCssClass(), dom);
};


/**
 * Takes a text caption or existing DOM structure, and returns it wrapped in
 * an appropriately-styled DIV.  Creates the following DOM structure:
 *    <div class="goog-inline-block goog-menu-button-caption">
 *      Contents...
 *    </div>
 * @param {goog.ui.ControlContent} content Text caption or DOM structure
 *     to wrap in a box.
 * @param {string} cssClass The CSS class for the renderer.
 * @param {goog.dom.DomHelper} dom DOM helper, used for document interaction.
 * @return {Element} Caption element.
 */
goog.ui.MenuButtonRenderer.wrapCaption = function(content, cssClass, dom) {
  return dom.createDom(
      'div',
      goog.ui.INLINE_BLOCK_CLASSNAME + ' ' +
          goog.getCssName(cssClass, 'caption'),
      content);
};


/**
 * Returns an appropriately-styled DIV containing a dropdown arrow element.
 * Creates the following DOM structure:
 *    <div class="goog-inline-block goog-menu-button-dropdown">
 *      &nbsp;
 *    </div>
 * @param {goog.dom.DomHelper} dom DOM helper, used for document interaction.
 * @return {Element} Dropdown element.
 */
goog.ui.MenuButtonRenderer.prototype.createDropdown = function(dom) {
  // 00A0 is &nbsp;
  return dom.createDom('div',
      goog.ui.INLINE_BLOCK_CLASSNAME + ' ' +
      goog.getCssName(this.getCssClass(), 'dropdown'), '\u00A0');
};


/**
 * Returns the CSS class to be applied to the root element of components
 * rendered using this renderer.
 * @return {string} Renderer-specific CSS class.
 * @override
 */
goog.ui.MenuButtonRenderer.prototype.getCssClass = function() {
  return goog.ui.MenuButtonRenderer.CSS_CLASS;
};
