(ns schedule
        (:import clojure.lang.IDeref
                 clojure.lang.Seqable
                 clojure.lang.Cons
                 java.util.Date
                 java.util.Calendar
                 java.util.TimeZone
                 java.io.Writer)
                                            )

(defprotocol Pattern
  (anchor  [pattern t]    "Anchors the scheduling pattern at time `t`, yielding a schedule starting after `t`"))

(defprotocol WeeklyTriggered
  (at-hour [pattern hour] "Constrains pattern, returning a new pattern that generates instants at hour `hour`"))

(defprotocol TimeZoneable
  (in-tz [timezonable tz] "Shifts timezoneable, returning a new object projected in timezoe `tz`."))

(defprotocol Instant
  (as-ts [instant] "Return a plaform native timestamp type corresponding to this instant.")
  (as-millis [instant] "Return a long representing the number of millis since the epoch corresponding to this instant."))

(defn current-time-millis
  []
        (System/currentTimeMillis)
                       
  )

      (extend-protocol Instant
        Date
        (as-ts [date] date)
        (as-millis [date] (.getTime date))
        Long
        (as-ts [long] (Date. long))
        (as-millis [long] long)
        Calendar
        (as-ts [cal] (.getTime cal))
        (as-millis [cal] (.getTimeInMillis cal)))

                               
                
                            
                                           
               
                                     
                               

(declare ^:private update-schedule-pattern)
(declare schedule-lazy-seq)

                      
                 
                 
                 
                 
                 
                 
                 
                   

(defn- calculate-next-schedule-instant
  [schedule]
        (let [pattern (.-pattern schedule)
              hour (.-hour pattern)
              tz (.-tz pattern)
              start (.-start schedule)
              cal (Calendar/getInstance)
              candidate-cal (Calendar/getInstance)]
          (.setTimeInMillis cal start)
          (.setTimeInMillis candidate-cal start)
          (.set candidate-cal Calendar/HOUR_OF_DAY hour)
          (.set candidate-cal Calendar/MINUTE 0)
          (.set candidate-cal Calendar/SECOND 0)
          (.set candidate-cal Calendar/MILLISECOND 0)
          (.setTimeZone candidate-cal (TimeZone/getTimeZone (or tz "UTC")))
          (if (< (.compareTo cal candidate-cal) 0)
            (.getTimeInMillis candidate-cal)
            (do (.add candidate-cal Calendar/DAY_OF_YEAR 1)
                (.getTimeInMillis candidate-cal))))
                                           
                                    
                                
                                       
                                 
                                                     
                                                                    
                                                                 
                                                                
                                                                         
                                                             
                                               
                                               
                                                  
                                                              
                                       
                                                                          
                                              )

(defn print-to-writer
  [val writer opts]
                                     
        (binding [*out* writer]
          (pr val)))

(defn print-weekly-schedule-to-abstract-writer
  [schedule writer write-fn opts]
  (let [pattern (.-pattern schedule)
        start (.-start schedule)]
    (write-fn writer "#schedule/weekly-schedule ")
    (print-to-writer {:pattern pattern
                      :start (as-ts start)} writer opts)))

(deftype WeeklySchedule [pattern start]
  Pattern
  (anchor  [this t]    (WeeklySchedule. pattern (as-millis t)))
  WeeklyTriggered
  (at-hour [this hour] (update-schedule-pattern at-hour hour))
  TimeZoneable
  (in-tz   [this tz]   (update-schedule-pattern in-tz tz))
        Seqable
        (seq [this] (schedule-lazy-seq this))
                 
                                               
                         
                                           
                                                                                  )

      (defmethod print-method WeeklySchedule
        [^WeeklySchedule schedule ^Writer w]
        (print-weekly-schedule-to-abstract-writer schedule w #(.write %1 %2) nil))

(defn schedule-lazy-seq
  [sched]
  (let [next-instant-ms (calculate-next-schedule-instant sched)
        pattern (.-pattern sched)]
    (cons (as-ts next-instant-ms)
          (lazy-seq (WeeklySchedule. pattern (inc next-instant-ms))))))

(defn- update-schedule-pattern
  [schedule f & args]
  (let [pattern (.-pattern schedule)
        start (.-start pattern)]
   (WeeklySchedule. (apply f pattern args) start)))


(defn- print-weekly-pattern-to-abstract-writer
  [pattern writer write-fn opts]
  (let [hour (.-hour pattern)
        tz (.-tz pattern)]
    (write-fn writer "#schedule/weekly-pattern \"Every day")
    (when hour
      (write-fn writer " at ")
      (write-fn writer (str hour))
      (write-fn writer ":00"))
    (when tz
      (write-fn writer " ")
      (write-fn writer tz))
    (write-fn writer "\"")))

(deftype WeeklyPattern [period hour tz]
  Pattern
  (anchor  [pattern t]    (WeeklySchedule. pattern (as-millis t)))
  WeeklyTriggered
  (at-hour [pattern new-hour] (WeeklyPattern. period new-hour tz))
  TimeZoneable
  (in-tz   [pattern new-tz]   (WeeklyPattern. period hour new-tz))
        IDeref
        (deref [pattern] (anchor pattern (current-time-millis)))
               
                                                                  
                         
                                          
                                                                                )

      (defmethod print-method WeeklyPattern
        [^WeeklyPattern pattern ^Writer w]
        (print-weekly-pattern-to-abstract-writer pattern w #(.write %1 %2) nil))

(defn read-schedule
  [m]
  (let [pattern (:pattern m)
        start (:start m)]
    (WeeklySchedule. pattern (as-millis start))))

(defn- string->int
  [int]
  (when int
          (Integer/parseInt int)
                               ))

(defn- floating-pattern
  [period hour tz]
  (WeeklyPattern. period (string->int hour) tz))

(defn- match-globs
  [re s]
        (let [matcher (re-matcher re s)]
          (re-find matcher))
                          )

(defn read-floating-pattern
        [s]
        (let [re #"Every day(?: at (\d+):00)?(?: (.+))?"
              [match hour tz] (match-globs re s)]
          (if match
            (floating-pattern nil hour tz)
            (throw (ex-info (str "Could not read floating pattern: \"" s \")
                            {:s s
                             :match match
                             :hour hour
                             :tz tz})))))

                                                                                    
                                                                             
;;;;;;;;;;;; This file autogenerated from cljx/src/schedule.cljx
