(ns com.vadelabs.workflow-core.main
  "Core Implementation"
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-exception.interface :as uex]
   [com.vadelabs.workflow-core.protocols :as wp]
   [com.vadelabs.workflow-core.spec :as ws]))

(defrecord Interceptor [enter leave error])

(extend-protocol wp/Interceptor
  #?(:clj clojure.lang.IPersistentMap
     :cljs cljs.core.PersistentHashMap)
  (interceptor [m] (map->Interceptor m))

  Interceptor
  (interceptor [r] r)

  #?(:clj clojure.lang.Fn :cljs function)
  (interceptor [f]
    (wp/interceptor {:enter f}))

  #?(:clj clojure.lang.Keyword
     :cljs cljs.core.Keyword)
  (interceptor [f]
    (wp/interceptor {:enter f}))

  #?(:bb sci.lang.Var
     :clj clojure.lang.Var
     :cljs cljs.core.Var)
  (interceptor [v]
    (wp/interceptor (deref v)))

  #?(:clj Object :cljs object)
  (interceptor [x]
    ;; Fallback : Could already be ILookup'able, would cover custom types (ex:
    ;; records)
    (when-not (instance? #?(:clj clojure.lang.ILookup
                            :cljs cljs.core.ILookup)
                x)
      (uex/raise
        :type :invalid
        :hint "Unsupported interceptor format/type"
        :val x)
      #_(throw (ex-info "Unsupported interceptor format/type"
                 {:exoscale.ex/type :exoscale.ex/invalid
                  :val x})))
    x))

;; Not working in cljs for some reason
#?(:clj
   (extend-protocol wp/Interceptor
     clojure.lang.Symbol
     (interceptor [s]
       (wp/interceptor (resolve s)))))

(def empty-queue (uc/queue))

(defn invoke-stage
  [ctx interceptor stage err]
  (if-let [f (get interceptor stage)]
    (try
      (let [ctx' (if err
                   (f (dissoc ctx ws/error) err)
                   (f ctx))]
        (cond-> ctx'
          (wp/async? ctx')
          (wp/catch (fn [e] (assoc ctx ws/error e)))))
      (catch #?(:clj Exception :cljs :default) e
        (assoc ctx ws/error e)))
    ctx))

(defn leave [ctx]
  (if (wp/async? ctx)
    (wp/then ctx leave)
    (let [stack (ws/stack ctx)]
      (if-let [interceptor (peek stack)]
        (recur (let [err (ws/error ctx)]
                 (invoke-stage (assoc ctx ws/stack (pop stack))
                   interceptor
                   (if err :error :leave)
                   err)))
        ctx))))

(defn enter [ctx]
  (if (wp/async? ctx)
    (wp/then ctx enter)
    (let [queue (ws/queue ctx)
          stack (ws/stack ctx)
          interceptor (peek queue)]
      (if (or (not interceptor)
            (ws/error ctx))
        ctx
        (-> (assoc ctx
              ws/queue (pop queue)
              ws/stack (conj stack interceptor))
          (invoke-stage interceptor :enter nil)
          recur)))))

(defn complete
  [ctx success error]
  (if (wp/async? ctx)
    (wp/then ctx #(complete % success error))
    (if-let [err (ws/error ctx)]
      (error err)
      (success ctx))))

(defn into-queue
  [q interceptors]
  (into (or q empty-queue)
    (map wp/interceptor)
    interceptors))

(defn enqueue
  [ctx interceptors]
  (update ctx
    ws/queue
    into-queue
    interceptors))

(defn execute
  [ctx success error]
  (-> ctx
    (enter)
    (leave)
    (complete success error)))
