(ns com.vadelabs.utils.blob
  (:require
    [com.vadelabs.utils.fressian :as u.fressian])
  (:import
    (java.io
      ByteArrayInputStream
      ByteArrayOutputStream
      DataInputStream
      DataOutputStream
      InputStream
      OutputStream)
    (net.jpountz.lz4
      LZ4Factory
      LZ4FrameInputStream
      LZ4FrameOutputStream)))


(set! *warn-on-reflection* true)

(def lz4-factory (LZ4Factory/fastestInstance))


(defn ^:private encode-v1
  [data]
  (with-open [^ByteArrayOutputStream output (ByteArrayOutputStream.)]
    (with-open [^DataOutputStream output (DataOutputStream. output)]
      (.writeShort output (short 5)) ;; version number
      (.writeInt output (int -1))
      (with-open [^OutputStream output (LZ4FrameOutputStream. output)]
        (-> (u.fressian/writer output)
          (u.fressian/write! data))))
    (.toByteArray output)))


(defn ^:private decode-v1
  [^bytes cdata]
  (with-open [^InputStream input (ByteArrayInputStream. cdata)]
    (.skip input 6)
    (with-open [^InputStream input (LZ4FrameInputStream. input)]
      (-> input u.fressian/reader u.fressian/read!))))


(defn encode
  ([data] (encode data nil))
  ([data {:keys [version]}]
    (let [version (or version 1)]
      (case version
        1 (encode-v1 data)
        (throw (ex-info "unsupported version" {:version version}))))))


(defn decode
  "A function used for decode persisted blobs in the database."
  [^bytes data]
  (with-open [bais (ByteArrayInputStream. data)
              dis  (DataInputStream. bais)]
    (let [version (.readShort dis)
          ulen    (.readInt dis)]
      (case version
        1 (decode-v1 data)
        (throw (ex-info "unsupported version" {:version version}))))))
