(ns com.vadelabs.utils.sql
  (:refer-clojure :exclude [name])
  (:require
    [com.vadelabe.utils.data :as u.data]
    [com.vadelabs.utils.str :as u.str]))


(defn dehyphen
  "Replace _embedded_ hyphens with spaces.
  Hyphens at the start or end of a string should not be touched."
  [s]
  (u.str/replace s #"(\w)-(?=\w)" "$1 "))


(defn literal
  [expr]
  (-> expr
    u.data/namify
    u.str/upper
    dehyphen))


(defn to-sql
  [arg]
  (cond
    (string? arg)
    [arg]
    (sequential? arg)
    arg))


(defn join
  [separator args]
  (let [args (map to-sql args)]
    (into []
      (cons (u.str/join separator
              (remove u.str/blank? (map first args)))
        (apply concat (map rest args))))))


(defn concat-sql
  [& args]
  (->> (remove nil? args)
    (map to-sql)
    (reduce (fn [stmt [sql & args]]
              (cons (apply str [(first stmt) sql])
                (concat (rest stmt) args)))
      [])))


(def ^:private quote-styles
  {:double-quote ["\"" "\""]
   :single-quote ["'" "'"]
   :backtick ["`" "`"]
   :square-bracket ["[" "]"]})


(def ^:private reserved
  #{"group"
    "cardinality"})


(defn name
  [{:keys [quote-style]} identifier]
  (let [partials (-> identifier u.data/namify (u.str/split #"\."))
        [start end] (get quote-styles quote-style)]
    (->> partials
      (map (fn [partial]
             (let [partial (if quote-style
                             partial
                             (u.str/replace partial #"-" "_"))]
               (if (contains? reserved partial)
                 (u.str/format "\"%s\"" partial)
                 (u.str/format "%s%s%s" start partial end)))))
      (u.str/join "."))))
