(ns com.vadelabs.utils.transit
  (:require
    [cognitect.transit :as t]
    [com.vadelabs.utils.data :as udata]
    [com.vadelabs.utils.uri :as uri]
    [lambdaisland.uri :as luri])
  #?(:clj
     (:import
       (java.io
         ByteArrayInputStream
         ByteArrayOutputStream
         File)
       java.time.OffsetDateTime
       lambdaisland.uri.URI)))


(def write-handlers (atom nil))
(def read-handlers (atom nil))
(def write-handler-map (atom nil))
(def read-handler-map (atom nil))


#?(:clj
   (defn str->bytes
     ([^String s]
       (str->bytes s "UTF-8"))
     ([^String s, ^String encoding]
       (.getBytes s encoding))))


#?(:clj
   (defn ^:private bytes->str
     ([^bytes data]
       (bytes->str data "UTF-8"))
     ([^bytes data, ^String encoding]
       (String. data encoding))))


(defn add-handlers!
  [& handlers]
  (letfn [(adapt-write-handler
            [{:keys [id class wfn]}]
            [class (t/write-handler (constantly id) wfn)])

          (adapt-read-handler
            [{:keys [id rfn]}]
            [id (t/read-handler rfn)])

          (merge-and-clean
            [m1 m2]
            (-> (merge m1 m2)
              (udata/without-nils)))]

    (let [rhs (into {}
                (comp
                  (filter :rfn)
                  (map adapt-read-handler))
                handlers)
          whs (into {}
                (comp
                  (filter :wfn)
                  (map adapt-write-handler))
                handlers)
          cwh (swap! write-handlers merge-and-clean whs)
          crh (swap! read-handlers merge-and-clean rhs)]

      (reset! write-handler-map #?(:clj (t/write-handler-map cwh) :cljs cwh))
      (reset! read-handler-map #?(:clj (t/read-handler-map crh) :cljs crh))
      nil)))


(add-handlers!
  #?(:clj
     {:id "file"
      :class File
      :wfn str
      :rfn identity})

  #?(:cljs
     {:id "n"
      :rfn (fn [value]
             (js/parseInt value 10))})
  #?(:cljs
     {:id "u"
      :rfn parse-uuid})

  #?(:clj
     {:id "m"
      :class OffsetDateTime
      :wfn (comp str inst-ms)})

  {:id "uri"
   :class #?(:clj URI :cljs luri/URI)
   :rfn uri/uri
   :wfn str})


#?(:clj
   (defn reader
     ([istream]
       (reader istream nil))
     ([istream {:keys [type] :or {type :json}}]
       (t/reader istream type {:handlers @read-handler-map}))))


#?(:clj
   (defn writer
     ([ostream]
       (writer ostream nil))
     ([ostream {:keys [type] :or {type :json}}]
       (t/writer ostream type {:handlers @write-handler-map}))))


#?(:clj
   (defn read!
     [reader]
     (t/read reader)))


#?(:clj
   (defn write!
     [writer data]
     (t/write writer data)))


#?(:clj
   (defn encode-str
     ([data] (encode-str data nil))
     ([data opts]
       (with-open [out (ByteArrayOutputStream.)]
         (t/write (writer out opts) data)
         (.toByteArray out)))))


#?(:clj
   (defn decode-str
     ([data] (decode-str data nil))
     ([data opts]
       (with-open [input (ByteArrayInputStream. ^bytes data)]
         (t/read (reader input opts))))))


(defn encode
  ([data] (encode data nil))
  ([data opts]
    #?(:cljs (let [t (:type opts :json)
                   w (t/writer t {:handlers @write-handler-map})]
               (t/write w data))
       :clj (->> (encode-str data opts)
              bytes->str))))


(defn decode
  ([data] (decode data nil))
  ([data opts]
    #?(:cljs
       (let [t (:type opts :json)
             r (t/reader t {:handlers @read-handler-map})]
         (t/read r data))
       :clj
       (-> (str->bytes data)
         (decode-str opts)))))


(defn transit?
  "Checks if a string can be decoded with transit"
  [v]
  (try
    (-> v decode nil? not)
    (catch #?(:cljs js/SyntaxError :clj Exception) _e
      false)))
