(ns com.vadelabs.utils.logging
  (:require
    [com.vadelabs.utils.data :as u.data]
    [com.vadelabs.utils.ex :as u.ex]
    [com.vadelabs.utils.id :as u.id]
    #?(:cljs [com.vadelabs.utils.pprint :as pp])
    [com.vadelabs.utils.str :as u.str]
    [promesa.exec :as px])
  #?(:cljs
     (:require-macros
       [com.vadelabs.utils.logging :as ulog]))
  #?(:clj
     (:import
       (org.slf4j
         Logger
         LoggerFactory))))


(def ^:dynamic *context* nil)

#?(:clj (set! *warn-on-reflection* true))


(defonce ^{:doc "A global log-record atom instance; stores last logged record."}
  log-record
  (atom nil))


(defonce
  ^{:doc "Default executor instance used for processing logs."
    :dynamic true}
  *default-executor*
  (delay
    #?(:clj  (px/single-executor :factory (px/thread-factory :name "vade/logger"))
       :cljs (px/microtask-executor))))


#?(:cljs
   (defonce loggers (js/Map.)))


#?(:cljs
   (declare level->int))


#?(:cljs
   (defn ^:private get-parent-logger
     [^string logger]
     (let [lindex (.lastIndexOf logger ".")]
       (.slice logger 0 (max lindex 0)))))


#?(:cljs
   (defn ^:private get-logger-level
     "Get the current level set for the specified logger. Returns int."
     [^string logger]
     (let [val (.get ^js/Map loggers logger)]
       (if (pos? val)
         val
         (loop [logger' (get-parent-logger logger)]
           (let [val (.get ^js/Map loggers logger')]
             (if (some? val)
               (do
                 (.set ^js/Map loggers logger val)
                 val)
               (if (= "" logger')
                 (do
                   (.set ^js/Map loggers logger 100)
                   100)
                 (recur (get-parent-logger logger'))))))))))


(defn enabled?
  "Check if logger has enabled logging for given level."
  [logger level]
  #?(:clj
     (let [logger (LoggerFactory/getLogger ^String logger)]
       (case level
         :trace (and (.isTraceEnabled ^Logger logger) logger)
         :debug (and (.isDebugEnabled ^Logger logger) logger)
         :info  (and (.isInfoEnabled  ^Logger logger) logger)
         :warn  (and (.isWarnEnabled  ^Logger logger) logger)
         :error (and (.isErrorEnabled ^Logger logger) logger)
         :fatal (and (.isErrorEnabled ^Logger logger) logger)
         (throw (IllegalArgumentException. (str "invalid level:"  level)))))
     :cljs
     (>= (level->int level)
       (get-logger-level logger))))


#?(:cljs
   (defn ^:private level->color
     [level]
     (case level
       :error "#c82829"
       :warn  "#f5871f"
       :info  "#4271ae"
       :debug "#969896"
       :trace "#8e908c")))


#?(:cljs
   (defn ^:private level->name
     [level]
     (case level
       :debug "DBG"
       :trace "TRC"
       :info  "INF"
       :warn   "WRN"
       :error "ERR")))


(defn level->int
  [level]
  (case level
    :trace 10
    :debug 20
    :info 30
    :warn 40
    :error 50))


(defn build-message
  [props]
  (loop [props  (seq props)
         result []]
    (if-let [[k v] (first props)]
      (if (simple-ident? k)
        (recur (next props)
          (conj result (str (name k) "=" (pr-str v))))
        (recur (next props)
          result))
      (u.str/join ", " result))))


(defn build-stack-trace
  [cause]
  #?(:clj  (u.ex/format-throwable cause)
     :cljs (.-stack ^js cause)))


#?(:cljs
   (defn ^:private get-special-props
     [props]
     (->> (seq props)
       (keep (fn [[k v]]
               (when (qualified-ident? k)
                 (cond
                   (= "js" (namespace k))
                   [:js (name k) (if (object? v) v (clj->js v))]

                   (= "error" (namespace k))
                   [:error (name k) v])))))))


(def ^:private reserved-props
  #{::level :cause ::logger ::sync? ::context})


(def ^:no-doc msg-props-xf
  (comp (partition-all 2)
    (map vec)
    (remove (fn [[k _]] (contains? reserved-props k)))))


(defn current-timestamp
  []
  #?(:clj (inst-ms (java.time.Instant/now))
     :cljs (js/Date.now)))


(defmacro log!
  "Emit a new log record to the global log-record state (asynchronously). "
  [& props]
  (let [{:keys [::level ::logger ::context ::sync? cause] :or {sync? false}} props
        props (into [] msg-props-xf props)]
    `(when (enabled? ~logger ~level)
       (let [props#   (cond-> (delay ~props) ~sync? deref)
             ts#      (current-timestamp)
             context# *context*
             logfn#   (fn []
                        (let [props#   (if ~sync? props# (deref props#))
                              props#   (into (u.data/ordered-map) props#)
                              cause#   ~cause
                              context# (u.data/without-nils
                                         (merge context# ~context))
                              lrecord# {::id (u.id/next)
                                        ::timestamp ts#
                                        ::message (delay (build-message props#))
                                        ::props props#
                                        ::context context#
                                        ::level ~level
                                        ::logger ~logger}
                              lrecord# (cond-> lrecord#
                                         (some? cause#)
                                         (assoc ::cause cause#
                                           ::trace (delay (build-stack-trace cause#))))]
                          (swap! log-record (constantly lrecord#))))]
         (if ~sync?
           (logfn#)
           (px/exec! *default-executor* logfn#))))))


#?(:clj
   (defn slf4j-log-handler
     {:no-doc true}
     [_ _ _ {:keys [::logger ::level ::trace ::message]}]
     (when-let [logger (enabled? logger level)]
       (let [message (cond-> @message
                       (some? trace)
                       (str "\n" @trace))]
         (case level
           :trace (.trace ^Logger logger ^String message)
           :debug (.debug ^Logger logger ^String message)
           :info  (.info  ^Logger logger ^String message)
           :warn  (.warn  ^Logger logger ^String message)
           :error (.error ^Logger logger ^String message)
           :fatal (.error ^Logger logger ^String message)
           (throw (IllegalArgumentException. (str "invalid level:"  level))))))))


#?(:cljs
   (defn console-log-handler
     {:no-doc true}
     [_ _ _ {:keys [::logger ::props ::level ::cause ::trace ::message]}]
     (when (enabled? logger level)
       (let [hstyles (u.str/ffmt "font-weight: 600; color: %" (level->color level))
             mstyles (u.str/ffmt "font-weight: 300; color: %" "#282a2e")
             header  (u.str/concat "%c" (level->name level) " [" logger "] ")
             message (u.str/concat header "%c" @message)]

         (js/console.group message hstyles mstyles)
         (doseq [[type n v] (get-special-props props)]
           (case type
             :js (js/console.log n v)
             :error (if (u.ex/error? v)
                      (js/console.error n (pr-str v))
                      (js/console.error n v))))

         (when cause
           (let [data    (ex-data cause)
                 explain (u.ex/explain data)]
             (when explain
               (js/console.log "Explain:")
               (js/console.log explain))

             (when (and data (not explain))
               (js/console.log "Data:")
               (js/console.log (pp/pprint-str data)))

             (js/console.log @trace #_(.-stack cause))))

         (js/console.groupEnd message)))))


#?(:clj  (add-watch log-record ::default slf4j-log-handler)
   :cljs (add-watch log-record ::default console-log-handler))


(defmacro set-level!
  "A CLJS-only macro for set logging level to current (that matches the
  current namespace) or user specified logger."
  ([level]
    (when (:ns &env)
      #_:clj-kondo/ignore
      `(.set ^js/Map loggers ~(str *ns*) (level->int ~level))))
  ([name level]
    (when (:ns &env)
      #_:clj-kondo/ignore
      `(.set ^js/Map loggers ~name (level->int ~level)))))


#?(:cljs
   (defn setup!
     [{:as config}]
     (run! (fn [[logger level]]
             (let [logger (if (keyword? logger) (name logger) logger)]
               (ulog/set-level! logger level)))
       config)))


(defmacro info
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :info ~@params)
     nil))


(defmacro inf
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :info ~@params)
     nil))


(defmacro error
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :error ~@params)
     nil))


(defmacro err
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :error ~@params)
     nil))


(defmacro warn
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :warn ~@params)
     nil))


(defmacro wrn
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :warn ~@params)
     nil))


(defmacro debug
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :debug ~@params)
     nil))


(defmacro dbg
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :debug ~@params)
     nil))


(defmacro trace
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :trace ~@params)
     nil))


(defmacro trc
  [& params]
  `(do
     (log! ::logger ~(str *ns*) ::level :trace ~@params)
     nil))
