(ns com.vadelabs.utils.ex
  (:require
    [clojure.spec.alpha :as s]
    [com.vadelabs.utils.pprint :as pp]
    [com.vadelabs.utils.schema :as sm]
    #?(:clj [com.vadelabs.utils.str :as ustr])
    [expound.alpha :as expound])
  #?(:cljs
     (:require-macros
       [com.vadelabs.utils.ex])))


#?(:clj (set! *warn-on-reflection* true))

(def incorrect ::incorrect)
(def unavailable ::unavailable)

(def interrupted ::interrupted)

(def forbidden ::forbidden)

(def unsupported ::unsupported)
(def not-found ::not-found)
(def conflict ::conflict)
(def fault ::fault)
(def busy ::busy)


(defmacro error
  [& {:keys [type hint] :as params}]
  `(ex-info ~(or hint (name type))
     (merge
       ~(dissoc params :cause ::data)
       ~(::data params))
     ~(:cause params)))


(defmacro raise
  [& params]
  `(throw (error ~@params)))


(defmacro ignoring
  [& exprs]
  (if (:ns &env)
    `(try ~@exprs (catch :default e# nil))
    `(try ~@exprs (catch Throwable e# nil))))


(defmacro try!
  [& exprs]
  (if (:ns &env)
    `(try ~@exprs (catch :default e# e#))
    `(try ~@exprs (catch Throwable e# e#))))


(defn ex-info?
  [v]
  (instance? #?(:clj clojure.lang.IExceptionInfo :cljs cljs.core.ExceptionInfo) v))


(defn error?
  [v]
  (instance? #?(:clj clojure.lang.IExceptionInfo :cljs cljs.core.ExceptionInfo) v))


(defn exception?
  [v]
  (instance? #?(:clj java.lang.Throwable :cljs js/Error) v))


#?(:clj
   (defn runtime-exception?
     [v]
     (instance? RuntimeException v)))


(defn explain
  ([data] (explain data nil))
  ([data {:keys [level length] :or {level 8 length 10}}]
    (cond

      (and (contains? data ::s/problems)
        (contains? data ::s/value)
        (contains? data ::s/spec))
      (binding [s/*explain-out* expound/printer]
        (with-out-str
          (s/explain-out (update data ::s/problems #(take length %)))))

      (contains? data ::sm/explain)
      (-> (sm/humanize-data (::sm/explain data))
        (pp/pprint-str {:level level :length length})))))


#?(:clj
   (defn format-throwable
     [^Throwable cause & {:keys [summary? detail? header? data? explain? chain? data-level data-length trace-length]
                          :or {summary? true
                               detail? true
                               header? true
                               data? true
                               explain? true
                               chain? true
                               data-length 10
                               data-level 8}}]

     (letfn [(print-trace-element
               [^StackTraceElement e]
               (let [class (.getClassName e)
                     method (.getMethodName e)
                     match (re-matches #"^([A-Za-z0-9_.-]+)\$(\w+)__\d+$" (str class))]
                 (if (and match (= "invoke" method))
                   (apply printf "%s/%s" (rest match))
                   (printf "%s.%s" class method)))
               (printf "(%s:%d)" (or (.getFileName e) "") (.getLineNumber e)))

             (print-explain
               [explain]
               (print "    xp: ")
               (let [[line & lines] (ustr/lines explain)]
                 (print line)
                 (newline)
                 (doseq [line lines]
                   (println "       " line))))

             (print-data
               [data]
               (when (seq data)
                 (print "    dt: ")
                 (let [[line & lines] (ustr/lines (pp/pprint-str data :level data-level :length data-length))]
                   (print line)
                   (newline)
                   (doseq [line lines]
                     (println "       " line)))))

             (print-trace-title
               [^Throwable cause]
               (print   " →  ")
               (printf "%s: %s" (.getName (class cause)) (first (ustr/lines (ex-message cause))))

               (when-let [^StackTraceElement e (first (.getStackTrace ^Throwable cause))]
                 (printf " (%s:%d)" (or (.getFileName e) "") (.getLineNumber e)))

               (newline))

             (print-summary
               [^Throwable cause]
               (let [causes (loop [cause (ex-cause cause)
                                   result []]
                              (if cause
                                (recur (ex-cause cause)
                                  (conj result cause))
                                result))]
                 (when header?
                   (println "SUMMARY:"))
                 (print-trace-title cause)
                 (doseq [cause causes]
                   (print-trace-title cause))))

             (print-trace
               [^Throwable cause]
               (print-trace-title cause)
               (let [st (.getStackTrace cause)]
                 (print "    at: ")
                 (if-let [e (first st)]
                   (print-trace-element e)
                   (print "[empty stack trace]"))
                 (newline)

                 (doseq [e (if (nil? trace-length) (rest st) (take (dec trace-length) (rest st)))]
                   (print "        ")
                   (print-trace-element e)
                   (newline))))

             (print-detail
               [^Throwable cause]
               (print-trace cause)
               (when-let [data (ex-data cause)]
                 (when data?
                   (print-data (dissoc data ::s/problems ::s/spec ::s/value ::sm/explain)))
                 (when explain?
                   (when-let [explain (explain data {:length data-length :level data-level})]
                     (print-explain explain)))))

             (print-all
               [^Throwable cause]
               (when summary?
                 (print-summary cause))

               (when detail?
                 (when header?
                   (println "DETAIL:"))

                 (print-detail cause)
                 (when chain?
                   (loop [cause cause]
                     (when-let [cause (ex-cause cause)]
                       (newline)
                       (print-detail cause)
                       (recur cause))))))]

       (with-out-str
         (print-all cause)))))


#?(:clj
   (defn print-throwable
     [cause & {:as opts}]
     (println (format-throwable cause opts))))
