(ns com.vadelabs.utils-core.anomaly
  (:refer-clojure :exclude [throw])
  (:require [clojure.spec.alpha :as s]))

(def ^:private info
  {::unavailable        {::retry-possible? true
                         ::fix :make-sure-callee-healthy}
   ::interrupted        {::retry-possible? true
                         ::fix :stop-interrupting}
   ::incorrect          {::retry-possible? false
                         ::fix :fix-caller-bug}
   ::forbidden          {::retry-possible? false
                         ::fix :fix-caller-creds}
   ::unsupported        {::retry-possible? false
                         ::fix :fix-caller-verb}
   ::not-found          {::retry-possible? false
                         ::fix :fix-caller-noun}
   ::conflict           {::retry-possible? false
                         ::fix :coordinate-with-callee}
   ::fault              {::retry-possible? false
                         ::fix :fix-callee-bug}
   ::busy               {::retry-possible? true
                         ::fix :backoff-and-retry}})

(def ^:private default-anom ::unavailable)

(s/def ::category #{::unavailable
                    ::interrupted
                    ::incorrect
                    ::forbidden
                    ::unsupported
                    ::not-found
                    ::conflict
                    ::fault
                    ::busy})

(s/def ::message string?)

(s/def ::id qualified-keyword?)

(s/def ::anomaly (s/keys :req [::category]
                   :opt [::message ::id]))

(defn ^:private merge-info [m]
  (let [category (or (::category m) default-anom)]
    (-> m
      (assoc ::category category)
      (merge (get info category)))))

(defn ->Anom
  ([]
   (->Anom nil nil))
  ([message]
   (->Anom message nil))
  ([message m]
   (merge-info (merge (if message {::message message} {}) m))))

(defn throw-anom
  ([]
   (throw-anom nil nil))
  ([message]
   (throw-anom message nil))
  ([message m]
   (throw (ex-info message (->Anom message m)))))

(defmacro ok->
  "When expr is not anomaly, threads it into the first form (via ->),
  and when that result is not an anomaly, through the next etc"
  [expr & forms]
  (let [g     (gensym)
        steps (map (fn [step] `(if (::category ~g) ~g (-> ~g ~step)))
                forms)]
    `(let [~g ~expr
           ~@(interleave (repeat g) (butlast steps))]
       ~(if (empty? steps)
          g
          (last steps)))))

(defmacro ok->>
  "When expr is not an anomaly, threads it into the first form (via ->>),
  and when that result is not an anomaly, through the next etc"
  [expr & forms]
  {:pre [(seq forms)]}
  (let [g     (gensym)
        steps (map (fn [step] `(if (::category ~g) ~g (->> ~g ~step)))
                forms)]
    `(let [~g ~expr
           ~@(interleave (repeat g) (butlast steps))]
       ~(if (empty? steps)
          g
          (last steps)))))

(defn anomaly? [x]
  (some? (::category x)))

(defn not-found? [x]
  (= ::not-found (::category x)))

(defn anomaly
  ([category message] (anomaly category message nil))
  ([category message context]
   (merge {::category category
           ::message  message}
     context)))

(defn throw-on-anomaly
  "Throw a exception if `x` is an anomaly, otherwise return it."
  [x]
  (if (anomaly? x)
    (throw (ex-info (::message x) x))
    x))

(defn anomaly!
  [& args]
  (->> args
    (apply anomaly)
    throw-on-anomaly))

(defn unavailable
  "Denotes an anomaly that can be fixed by
  making sure the callee is healthy.

  Can be meaningfully retried callee-side."
  ([message] (unavailable message nil))
  ([message context] (anomaly ::unavailable message context)))

(defn unavailable!
  [& args]
  (->> args
    (apply unavailable)
    throw-on-anomaly))

(defn interrupted
  "Denotes an anomaly that can be fixed by
  refraining from interrupting the callee.

  Can be meaningfully retried callee-side."
  ([message] (interrupted message nil))
  ([message context] (anomaly ::interrupted message context)))

(defn interrupted!
  [& args]
  (->> args
    (apply interrupted)
    throw-on-anomaly))

(defn incorrect
  "Denotes an anomaly that can be fixed by
  fixing a bug caller-side.

  Can't be meaningfully (immediately) retried callee-side."
  ([message] (incorrect message nil))
  ([message context] (anomaly ::incorrect message context)))

(defn incorrect!
  [& args]
  (->> args
    (apply incorrect)
    throw-on-anomaly))

(defn forbidden
  "Denotes an anomaly that can be fixed by
  fixing caller creds.

  Can't be meaningfully (immediately) retried callee-side."
  ([message] (forbidden message nil))
  ([message context] (anomaly ::forbidden message context)))

(defn forbidden!
  [& args]
  (->> args
    (apply forbidden)
    throw-on-anomaly))

(defn unsupported
  "Denotes an anomaly that can be fixed by
  fixing the caller's verb.

  Can't be meaningfully (immediately) retried callee-side."
  ([message] (unsupported message nil))
  ([message context] (anomaly ::unsupported message context)))

(defn unsupported!
  [& args]
  (->> args
    (apply unsupported)
    throw-on-anomaly))

(defn not-found
  "Denotes an anomaly that can be fixed by
  fixing the passed noun.

  Can't be meaningfully (immediately) retried callee-side."
  ([message] (not-found message nil))
  ([message context] (anomaly ::not-found message context)))

(defn not-found!
  [& args]
  (->> args
    (apply not-found)
    throw-on-anomaly))

(defn conflict
  "Denotes an anomaly that can be fixed by
  coordinating with the callee.

  Can't be meaningfully (immediately) retried callee-side."
  ([message] (conflict message nil))
  ([message context] (anomaly ::conflict message context)))

(defn conflict!
  [& args]
  (->> args
    (apply conflict)
    throw-on-anomaly))

(defn fault
  "Denotes an anomaly that can be fixed by
  fixing a bug callee-side.

  Can't be meaningfully (immediately) retried callee-side."
  ([message] (fault message nil))
  ([message context] (anomaly ::fault message context)))

(defn fault!
  [& args]
  (->> args
    (apply fault)
    throw-on-anomaly))

(defn busy
  "Denotes an anomaly that can be fixed by
  backing off and retrying callee-side.

  Can be meaningfully retried callee-side."
  ([message] (busy message nil))
  ([message context] (anomaly ::busy message context)))

(defn busy!
  [& args]
  (->> args
    (apply busy)
    throw-on-anomaly))

(defn ->anomaly
  "Returns just the anomaly fields from a map"
  [m]
  (select-keys m [::category ::message]))

(defn dissoc-anomaly
  [m]
  (dissoc m ::category ::message))

(defn category
  [m]
  (::category m))

(defn message
  [m]
  (::message m))

(def http-status->message
  {100 "Continue"
   101 "Switching Protocols"
   102 "Processing"
   200 "OK"
   201 "Created"
   202 "Accepted"
   203 "Non-Authoritative Information"
   204 "No Content"
   205 "Reset Content"
   206 "Partial Content"
   207 "Multi-Status"
   208 "Already Reported"
   226 "IM Used"
   300 "Multiple Choices"
   301 "Moved Permanently"
   302 "Found"
   303 "See Other"
   304 "Not Modified"
   305 "Use Proxy"
   306 "Reserved"
   307 "Temporary Redirect"
   308 "Permanent Redirect"
   400 "Bad Request"
   401 "Unauthorized"
   402 "Payment Required"
   403 "Forbidden"
   404 "Not Found"
   405 "Method Not Allowed"
   406 "Not Acceptable"
   407 "Proxy Authentication Required"
   408 "Request Timeout"
   409 "Conflict"
   410 "Gone"
   411 "Length Required"
   412 "Precondition Failed"
   413 "Request Entity Too Large"
   414 "Request-URI Too Long"
   415 "Unsupported Media Type"
   416 "Requested Range Not Satisfiable"
   417 "Expectation Failed"
   422 "Unprocessable Entity"
   423 "Locked"
   424 "Failed Dependency"
   425 "Unassigned"
   426 "Upgrade Required"
   427 "Unassigned"
   428 "Precondition Required"
   429 "Too Many Requests"
   430 "Unassigned"
   431 "Request Header Fields Too Large"
   500 "Internal Server Error"
   501 "Not Implemented"
   502 "Bad Gateway"
   503 "Service Unavailable"
   504 "Gateway Timeout"
   505 "HTTP Version Not Supported"
   506 "Variant Also Negotiates (Experimental)"
   507 "Insufficient Storage"
   508 "Loop Detected"
   509 "Unassigned"
   510 "Not Extended"
   511 "Network Authentication Required"})

(def http-status->anomaly
  {400 ::incorrect   ;; Bad Request [RFC7231, Section 6.5.1]
   401 ::forbidden   ;; Unauthorized [RFC7235, Section 3.1]
   402 ::forbidden   ;; Payment Required [RFC7231, Section 6.5.2]
   403 ::forbidden   ;; Forbidden [RFC7231, Section 6.5.3]
   404 ::not-found   ;; Not Found [RFC7231, Section 6.5.4]
   405 ::unsupported ;; Method Not Allowed [RFC7231, Section 6.5.5]
   406 ::unsupported ;; Not Acceptable [RFC7231, Section 6.5.6]
   407 ::forbidden   ;; Proxy Authentication Required [RFC7235, Section 3.2]
   408 ::unavailable ;; Request Timeout [RFC7231, Section 6.5.7]
   409 ::conflict    ;; Conflict [RFC7231, Section 6.5.8]
   410 ::not-found   ;; Gone [RFC7231, Section 6.5.9]
   411 ::incorrect   ;; Length Required [RFC7231, Section 6.5.10]
   412 ::unavailable ;; ?? ;; Precondition Failed [RFC7232, Section 4.2][RFC8144, Section 3.2]
   413 ::incorrect   ;; Payload Too Large [RFC7231, Section 6.5.11]
   414 ::incorrect   ;; URI Too Long [RFC7231, Section 6.5.12]
   415 ::unsupported ;; Unsupported Media Type [RFC7231, Section 6.5.13][RFC7694, Section 3]
   416 ::incorrect   ;; Range Not Satisfiable [RFC7233, Section 4.4]
   417 ::unsupported ;; Expectation Failed [RFC7231, Section 6.5.14]
   418 ::unsupported ;; teapot
                  ;;419  ;;Unassigned
                  ;;420  ;;Unassigned
   421 ::incorrect   ;; Misdirected Request [RFC7540, Section 9.1.2]
   422 ::incorrect   ;; Unprocessable Entity [RFC4918]
   423 ::unavailable ;; Locked [RFC4918]
   424 ::unavailable ;; Failed Dependency [RFC4918]
                  ;;425 ;; Unassigned
   426 ::incorrect   ;; Upgrade Required [RFC7231, Section 6.5.15]
                  ;;427 ;; Unassigned
   428 ::incorrect   ;; Precondition Required [RFC6585]
   429 ::busy        ;; Too Many Requests [RFC6585]
                  ;;430 ;; Unassigned
   431 ::unsupported ;; Request Header Fields Too Large [RFC6585]
   451 ::unavailable ;; Unavailable For Legal Reasons [RFC7725]
   500 ::fault       ;; Internal Server Error [RFC7231, Section 6.6.1]
   501 ::unsupported ;; Not Implemented [RFC7231, Section 6.6.2]
   502 ::unavailable ;; Bad Gateway [RFC7231, Section 6.6.3]
   503 ::unavailable ;; Service Unavailable [RFC7231, Section 6.6.4]
   504 ::unavailable ;; Gateway Timeout [RFC7231, Section 6.6.5]
   505 ::unsupported ;; HTTP Version Not Supported [RFC7231, Section 6.6.6]
   506 ::fault       ;; Variant Also Negotiates [RFC2295]
   507 ::unavailable ;; Insufficient Storage [RFC4918]
   508 ::fault       ;; Loop Detected [RFC5842]
                  ;;509 ;; Unassigned
   510 ::fault       ;; Not Extended [RFC2774]
   511 ::forbidden   ;; Network Authentication Required [RFC6585]
   })
