(ns ^:no-doc com.vadelabs.turbo-ui.explorer.ui.app
  (:require ["react" :as react]
            [clojure.string :as str]
            [com.vadelabs.turbo-ui.explorer.colors :as c]
            [com.vadelabs.turbo-ui.explorer.ui.api :as api]
            [com.vadelabs.turbo-ui.explorer.ui.commands :as commands]
            [com.vadelabs.turbo-ui.explorer.ui.connecton-status :as status]
            [com.vadelabs.turbo-ui.explorer.ui.drag-and-drop :as dnd]
            [com.vadelabs.turbo-ui.explorer.ui.icons :as icons]
            [com.vadelabs.turbo-ui.explorer.ui.inspector :as ins]
            [com.vadelabs.turbo-ui.explorer.ui.options :as opts]
            [com.vadelabs.turbo-ui.explorer.ui.select :as select]
            [com.vadelabs.turbo-ui.explorer.ui.state :as state]
            [com.vadelabs.turbo-ui.explorer.ui.styled :as s]
            [com.vadelabs.turbo-ui.explorer.ui.theme :as theme]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.bin :as bin]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.charts :as charts]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.cljdoc :as cljdoc]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.code :as code]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.csv :as csv]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.date-time :as date-time]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.deref :as deref]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.diff :as diff]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.edn :as edn]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.exception :as ex]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.hiccup :as hiccup]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.html :as html]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.http :as http]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.image :as image]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.json :as json]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.jwt :as jwt]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.log :as log]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.markdown :as md]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.pprint :as pprint]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.prepl :as prepl]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.relative-time :as relative-time]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.table :as table]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.test-report :as test-report]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.text :as text]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.transit :as transit]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.tree :as tree]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.vega :as vega]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.vega-lite :as vega-lite]
            [com.vadelabs.turbo-ui.explorer.ref :as pref]
            [reagent.core :as r]
            [com.vadelabs.turbo-css.interface :refer [sx]]))

(defn- selected-context-view []
  (let [theme (theme/use-theme)
        state (state/use-state)
        path  (state/get-path @state)]
    [s/div
     {#_:style
      #_{:max-width "100vw"
         :display :flex
         :align-items :center
         :justify-content :space-between
         :background (::c/background2 theme)
         :color (::c/text theme)
         :border-top [1 :solid (::c/border theme)]}}
     [s/div
      {:title "Copy path."
       :on-click #(commands/copy-path state)
       :style/hover {:color (::c/tag theme)}
       :style
       {:color (::c/border theme)
        :cursor :pointer
        :box-sizing :border-box
        :padding (:padding theme)
        :border-right [1 :solid (::c/border theme)]}}
      [icons/copy]]
     [s/div
      {:style
       {:cursor :pointer
        :overflow :auto
        :display :flex
        :box-sizing :border-box
        :padding (:padding theme)
        :gap (:padding theme)}}
      [s/div {:style {:grid-row "1"}} "["]
      (map-indexed
        (fn [idx k]
          ^{:key idx}
          [s/div {:style {:grid-row "1"}} [ins/preview k]])
        path)
      [s/div {:style {:grid-row "1"}} "]"]]
     [s/div {:style {:flex "1"}}]]))

(defn- runtime-info []
  (let [opts       (opts/use-options)
        theme      (theme/use-theme)
        connected? (status/use-status)
        header     (if (= :dev (:mode opts))
                     (::c/diff-add theme)
                     (::c/background2 theme))]
    #_(react/useEffect
        (fn []
          (state/set-theme header)
          (state/notify-parent {:type :set-theme :color header}))
        #js [header])
    #_(react/useEffect
        (fn []
          (when-let [{:keys [name platform version]} opts]
            (state/set-title!
              (str/join
                " - "
                [(:window-title opts name) platform version]))))
        #js [opts])
    (when-not connected?
      [s/div
       {:style {:color (::c/exception theme)}}
       "ERROR: Disconnected from runtime!"])))

(defn inspect-footer []
  (let [theme (theme/use-theme)
        state (state/use-state)
        selected-context (state/get-selected-context @state)
        viewer           (ins/get-viewer state selected-context)
        compatible-viewers (ins/get-compatible-viewers @ins/viewers selected-context)]
    [s/div
     {:style
      {:display :flex
       :padding-top (:padding theme)
       :padding-bottom (:padding theme)
       :padding-right (* 1.5 (:padding theme))
       :padding-left (* 1.5 (:padding theme))
       :align-items :stretch
       :justify-content :space-between
       :background (::c/background2 theme)
       :box-sizing :border-box
       :border-top [1 :solid (::c/border theme)]}}
     [s/select
      {:title "Select a different viewer."
       :value (pr-str (:name viewer))
       :on-change
       (fn [e]
         (ins/set-viewer!
           state
           selected-context
           (keyword (.substr (.. e -target -value) 1))))
       :style
       {:background (::c/background theme)
        :padding (:padding theme)
        :box-sizing :border-box
        :font-family (:font-family theme)
        :font-size (:font-size theme)
        :color (::c/text theme)
        :border-radius (:border-radius theme)
        :border [1 :solid (::c/border theme)]}}
      (for [{:keys [name]} compatible-viewers]
        ^{:key name}
        [s/option {:value (pr-str name)} (pr-str name)])]
     [runtime-info]
     [s/button
      {:title    "Open command palette."
       :on-click #(commands/open-command-palette state)
       :style
       {:font-family (:font-family theme)
        :background (::c/background theme)
        :border-radius (:border-radius theme)
        :border [1 :solid (::c/border theme)]
        :box-sizing :border-box
        :padding-top (:padding theme)
        :padding-bottom (:padding theme)
        :padding-left (* 2.5 (:padding theme))
        :padding-right (* 2.5 (:padding theme))
        :color (::c/tag theme)
        :font-size (:font-size theme)
        :font-weight :bold
        :cursor :pointer}}
      [icons/terminal]]]))

(defn- search-input []
  (let [ref      (react/useRef nil)
        theme    (theme/use-theme)
        state    (state/use-state)
        context  (state/get-selected-context @state)
        location (state/get-location context)
        color    (if-let [depth (:depth context)]
                   (nth theme/order depth)
                   ::c/border)]
    (react/useEffect
      (fn []
        (swap! commands/search-refs conj ref)
        #(swap! commands/search-refs disj ref)))
    [s/input
     {:ref ref
      :disabled  (nil? context)
      :on-change #(let [value (.-value (.-target %))]
                    (when context
                      (state/dispatch!
                        state
                        update
                        :search-text
                        (fn [filters]
                          (if (str/blank? value)
                            (dissoc filters location)
                            (assoc filters location value))))))
      :on-key-down (fn [e]
                     (when (= (.-key e) "Enter")
                       (.blur (.-current ref))))
      :value (get-in @state [:search-text location] "")
      :placeholder (if-not context
                     "Select a value to enable filtering"
                     "Type here to begin filtering")
      :style
      {:background (::c/background theme)
       :padding (:padding theme)
       :box-sizing :border-box
       :font-family (:font-family theme)
       :font-size (:font-size theme)
       :color (get theme color)
       :border [1 :solid (::c/border theme)]
       :border-radius (:border-radius theme)}
      :style/placeholder
      {:color (if-not context (::c/border theme) (::c/text theme))}}]))

(defn- button-hover [props child]
  (let [theme (theme/use-theme)]
    [s/div
     (merge
       {:style {:display         :flex
                :width           "2rem"
                :height          "2rem"
                :border-radius   "100%"
                :align-items     :center
                :justify-content :center
                :cursor          :pointer
                :border          [1 :solid "rgba(0,0,0,0)"]}}
       props
       (when-not (:disabled props)
         {:style/hover
          {:background (::c/background theme)
           :border     [1 :solid (::c/border theme)]}}))
     child]))

(defn- toolbar []
  (let [theme (theme/use-theme)
        state (state/use-state)]
    [s/div
     {:style
      {:display :grid
       :grid-template-columns "auto auto 1fr auto"
       :box-sizing :border-box
       :padding-top (:padding theme)
       :padding-bottom (:padding theme)
       :padding-right (* 1.5 (:padding theme))
       :padding-left (* 1.5 (:padding theme))
       :grid-gap (* 1.5 (:padding theme))
       :background (::c/background2 theme)
       :align-items :center
       :justify-content :center
       :border-top [1 :solid (::c/border theme)]
       :border-bottom [1 :solid (::c/border theme)]}}
     (let [disabled? (nil? @(r/cursor state [:portal/previous-state]))]
       [button-hover
        {:disabled disabled?
         :on-click #(state/dispatch! state state/history-back)}
        [icons/arrow-left {:disabled disabled?
                           :title    "Go back in portal history."
                           :size     "2x"
                           :style    (merge
                                       {:transform     "scale(0.75)"
                                        :color         (::c/text theme)}
                                       (when disabled?
                                         {:opacity 0.45
                                          :cursor  :default}))}]])
     (let [disabled? (nil? @(r/cursor state [:portal/next-state]))]
       [button-hover
        {:disabled disabled?
         :on-click #(state/dispatch! state state/history-forward)}
        [icons/arrow-right {:disabled disabled?
                            :title    "Go forward in portal history."
                            :size     "2x"
                            :style    (merge
                                        {:transform     "scale(0.75)"
                                         :color         (::c/text theme)}
                                        (when disabled?
                                          {:opacity 0.45
                                           :cursor  :default}))}]])
     [search-input]
     [button-hover
      {:on-click #(state/dispatch! state state/clear)}
      [icons/ban
       {:title    "Clear all values from com.vadelabs.turbo-ui.explorer. - CTRL L"
        :size     "2x"
        :style    (merge
                    {:transform     "scale(0.75)"
                     :color         (::c/text theme)})}]]]))

(defn inspect-1 [value]
  (let [theme (theme/use-theme)
        state (state/use-state)
        ref   (react/useRef)]
    (react/useEffect
      (fn []
        (when-let [el (.-current ref)]
          (state/dispatch! state assoc :scroll-element el))
        (fn []))
      #js [(.-current ref)])
    [s/div
     {:style
      {:height "100vh"
       :display :flex
       :flex-direction :column}}
     (when-not (:disable-shell? @state) [toolbar])
     [s/div
      {:style
       {:flex "1"
        :position :relative
        :min-width "100%"
        :box-sizing :border-box}}
      [s/div
       {:ref ref
        :on-click #(state/dispatch! state state/clear-selected)
        :style
        {:position :absolute
         :top 0
         :left 0
         :right 0
         :bottom 0
         :overflow :auto
         :box-sizing :border-box}}
       [s/div
        {:style {:min-width :fit-content}}
        [s/div
         {:style
          {:min-width :fit-content
           :box-sizing :border-box
           :padding (* 2 (:padding theme))}}
         [:> ins/error-boundary
          [select/with-position
           {:row 0 :column 0}
           [ins/inspector value]]]]]]]
     (when-not (:disable-shell? @state)
       [:<>
        #_[inspect-footer]
        [selected-context-view]])]))

(defn scrollbars []
  (when-not (theme/is-vs-code?)
    (let [thumb "rgba(0,0,0,0.3)"]
      [:style
       (str "* { scrollbar-color: " thumb " rgba(0,0,0,0); } "
         "*::-webkit-scrollbar { width: 10px; height: 10px; }"
         "*::-webkit-scrollbar-corner { opacity: 0 }"
         "*::-webkit-scrollbar-track  { opacity: 0 }"
         "*::-webkit-scrollbar-thumb  { background-color: " thumb "; }"
         "*::-webkit-scrollbar-thumb  { border-radius: 10px; }")])))

(defn text-selection []
  (let [style "background: rgba(0,0,0,0.5)"]
    [:style
     (str "::selection { " style " }")
     (str "::-moz-selection { " style " }")]))

(defn styles []
  [:<>
   [vega/styles]
   [prepl/styles]
   [hiccup/styles]
   [text-selection]
   [code/stylesheet]])

(defn- container [children]
  (let [theme (theme/use-theme)]
    (into
      [s/div
       {:class (sx)
        :style
        {;; :-webkit-app-region (when-not (theme/is-vs-code?) :drag)
        ;; :display :flex
        ;; :flex-direction :column
         :background (::c/background theme)
         :color (::c/text theme)
         :font-family (:font-family theme)
         :font-size (:font-size theme)
        ;; :height "100%"
        ;; :width "100%"
         }}
       [styles]
       [scrollbars]]
      children)))

(defn- inspect-1-history [default-value]
  (let [current-state @(state/use-state)]
    [:<>
     [commands/palette]
     (doall
       (map-indexed
         (fn [index state]
           ^{:key index}
           [s/div
            {:style
             {:flex "1"
              :display
              (if (= state current-state)
                :block
                :none)}}
            [select/with-position
             {:row 0 :column index}
             [inspect-1 (state/get-value state default-value)]]])
         (state/get-history current-state)))]))

(def viewers
  ^{:com.vadelabs.turbo-ui.explorer.viewer/default :com.vadelabs.turbo-ui.explorer.viewer/table
    :com.vadelabs.turbo-ui.explorer.viewer/table {:columns [:name :doc]}}
  [ex/viewer
   ex/trace-viewer
   charts/line-chart
   charts/scatter-chart
   charts/histogram-chart
   log/viewer
   http/viewer
   image/viewer
   deref/viewer
   test-report/viewer
   prepl/viewer
   cljdoc/viewer
   ins/viewer
   pprint/viewer
   vega-lite/viewer
   vega/viewer
   bin/viewer
   table/viewer
   tree/viewer
   text/viewer
   code/viewer
   code/pr-str-viewer
   json/viewer
   jwt/viewer
   edn/viewer
   transit/viewer
   csv/viewer
   html/viewer
   diff/viewer
   md/viewer
   hiccup/viewer
   date-time/viewer
   relative-time/viewer])

(reset! api/viewers viewers)

(defn root [& children]
  (let [opts  (opts/use-options)
        state state/state
        theme (or (:theme @state)
                (:theme opts)
                (::c/theme opts))]
    [state/with-state
     state
     [theme/with-theme
      theme
      [dnd/area
       [container children]]]]))

(defn app [value]
  ;; (println value "value>>>")
  [root
   [s/div
    {:style
     {;; :width "100%"
      ;; :height "100%"
      :display :flex}}
    [inspect-1-history value]]])
