(ns com.vadelabs.turbo-core.compiler.aot
  "Compiler code that translates HyperScript into React calls at compile-time."
  (:require
   [cljs.analyzer :as ana]
   [com.vadelabs.turbo-core.compiler.attributes :as attrs]
   [com.vadelabs.turbo-core.compiler.js :as js]
   [com.vadelabs.turbo-core.utils :as utils]))

(defmulti compile-attrs
  "Compiles a map of attributes into JS object,
  or emits interpretation call for runtime, when a value
  at props position is dynamic (symbol)"
  (fn [tag attrs opts] tag))

(defmethod compile-attrs :element [_ attrs {:keys [tag-id-class]}]
  (if (or (map? attrs) (nil? attrs))
    `(cljs.core/array
       ~(cond-> attrs
         ;; interpret :style if it's not map literal
          (and (some? (:style attrs))
            (not (map? (:style attrs))))
          (assoc :style `(com.vadelabs.turbo-core.compiler.attributes/convert-props ~(:style attrs) (cljs.core/array) true))
         ;; merge parsed id and class with attrs map
          :always (attrs/set-id-class tag-id-class)
         ;; camel-casify the map
          :always (attrs/compile-attrs {:custom-element? (last tag-id-class)})
         ;; emit JS object literal
          :always js/to-js))
    ;; otherwise emit interpretation call
    `(com.vadelabs.turbo-core.compiler.attributes/interpret-attrs ~attrs (cljs.core/array ~@tag-id-class) false)))

(defmethod compile-attrs :component [_ props _]
  (if (or (map? props) (nil? props))
    `(cljs.core/array ~props)
    `(com.vadelabs.turbo-core.compiler.attributes/interpret-props ~props)))

(defmethod compile-attrs :fragment [_ attrs _]
  (if (map? attrs)
    `(cljs.core/array ~(-> attrs attrs/compile-attrs js/to-js))
    `(com.vadelabs.turbo-core.compiler.attributes/interpret-attrs ~attrs (cljs.core/array) false)))

(defn ^:private input-component? [x]
  (contains? #{"input" "textarea"} x))

(defn ^:private turbo-element?
  "Returns true when `form` is `(com.vadelabs.turbo-core.interface/$ ...)`"
  [env form]
  (and (list? form)
    (symbol? (first form))
    (->> (first form) (ana/resolve-var env) :name (= 'com.vadelabs.turbo-core.interface/$))))

(def elements-list-fns
  '#{for map mapv filter filterv remove keep keep-indexed})

(defn ^:private elements-list?
  "Returns true when `v` is form commonly used to render a list of elements
  `(map ...)`, `(for ...)`, etc"
  [v]
  (and (list? v)
    (symbol? (first v))
    (elements-list-fns (first v))))

(defn ^:private normalize-element
  "When the second item in the element `v` is either Turbo element or `elements-list?`,
  returns normalized element with empty map at props position
  and child element shifted into children position"
  [env v]
  (if (or (turbo-element? env (second v))
        (elements-list? (second v)))
    (into [(first v) {}] (rest v))
    v))

(defmulti compile-element*
  "Compiles turbo elements into React.createElement"
  (fn [[tag] _]
    (cond
      (= :<> tag) :fragment
      (keyword? tag) :element
      :else :component)))

(defmethod compile-element* :element [v {:keys [env]}]
  (let [[tag attrs & children] (normalize-element env v)
        tag-id-class (attrs/parse-tag tag)
        attrs-children (compile-attrs :element attrs {:tag-id-class tag-id-class})
        tag-str (first tag-id-class)
        ret (if (input-component? tag-str)
              `(create-turbo-input ~tag-str ~attrs-children (cljs.core/array ~@children))
              `(>el ~tag-str ~attrs-children (cljs.core/array ~@children)))]
    ret))

(defmethod compile-element* :component [v {:keys [env]}]
  (let [[tag props & children] (normalize-element env v)
        tag (vary-meta tag assoc :tag 'js)
        props-children (compile-attrs :component props nil)]
    `(com.vadelabs.turbo-core.compiler/component-element ~tag ~props ~props-children (cljs.core/array ~@children))))

(defmethod compile-element* :fragment [v _]
  (let [[_ attrs & children] v
        attrs (compile-attrs :fragment attrs nil)
        ret `(>el fragment ~attrs (cljs.core/array ~@children))]
    ret))

(defn compile-element [v {:keys [env] :as opts}]
  (if (utils/cljs-env? env)
    (compile-element* v opts)
    v))
