(ns com.vadelabs.turbo-ui.explorer.ui.rpc
  (:refer-clojure :exclude [read type])
  (:require [com.vadelabs.turbo-ui.explorer.async :as a]
            [com.vadelabs.turbo-ui.explorer.runtime.cson :as cson]
            [com.vadelabs.turbo-ui.explorer.ui.cljs :as cljs]
            [com.vadelabs.turbo-ui.explorer.ui.rpc.runtime :as rt]
            [com.vadelabs.turbo-ui.explorer.ui.state :as state]
            [com.vadelabs.turbo-ui.explorer.ui.viewer.diff :as diff])
  (:import [goog.math Long]))

(defn call [f & args]
  (apply state/invoke f args))

(when (exists? js/BigInt)
  (extend-type js/BigInt
    IHash
    (-hash [this]
      (hash (.toString this)))
    IPrintWithWriter
    (-pr-writer [this writer _opts]
      (-write writer (str this "N")))))

(when (exists? js/Uint8Array)
  (extend-type js/Uint8Array
    IPrintWithWriter
    (-pr-writer [this writer _opts]
      (-write writer "#portal/bin \"")
      (-write writer (cson/base64-encode this))
      (-write writer "\""))))

(extend-type array
  IHash
  (-hash [this]
    (hash (into [] this)))
  IEquiv
  (-equiv [^js this ^js o]
    (let [n (.-length this)]
      (and (array? o)
        (= n (.-length o))
        (loop [i 0]
          (cond
            (== i n)             true
            (not= (aget this i)
              (aget this o)) false
            :else                (recur (inc i))))))))

(extend-type number
  IEquiv
  (-equiv [a b]
    (or (== a b)
      (and (.isNaN js/Number a)
        (.isNaN js/Number b)))))

(extend-type Long
  IPrintWithWriter
  (-pr-writer [this writer _opts]
    (-write writer (str this))))

(defmethod cson/tagged-str "remote" [{:keys [rep]}] rep)

(when-not js/goog.DEBUG
  (extend-type default
    cson/ToJson
    (-to-json [value buffer]
      (cson/-to-json
        (with-meta
          (cson/tagged-value "remote" (pr-str value))
          (meta value))
        buffer))))

(defn- read [string]
  (cson/read
    string
    {:transform rt/transform
     :default-handler
     (fn [op value]
       (or (case op
             "ref"    (rt/->value value)
             "object" (rt/->object call value)
             (diff/->diff op value))
         (cson/tagged-value op value)))}))

(defn- write [value]
  (cson/write
    value
    {:transform
     (fn [value]
       (if-let [id (-> value meta :com.vadelabs.turbo-ui.explorer.runtime/id)]
         (cson/tagged-value "ref" id)
         value))}))

(defonce ^:private id (atom 0))
(defonce ^:private pending-requests (atom {}))

(defn- next-id [] (swap! id inc))

(declare send!)

(defn- ws-request [message]
  (js/Promise.
    (fn [resolve reject]
      (let [id (:com.vadelabs.turbo-ui.explorer.rpc/id message)]
        (swap! pending-requests assoc id [resolve reject])
        (send! (assoc message :com.vadelabs.turbo-ui.explorer.rpc/id id))))))

(defn- web-request [message]
  (js/Promise.
    (fn [resolve reject]
      (try
        (-> (write message)
          js/window.opener.com.vadelabs.turbo-ui.explorer.web.send_BANG_
          (.then read)
          resolve)
        (catch :default e (reject e))))))

(defn request [message]
  ((if js/window.opener web-request ws-request)
   (assoc message :com.vadelabs.turbo-ui.explorer.rpc/id (next-id))))

(def ^:private ops
  {:com.vadelabs.turbo-ui.explorer.rpc/response
   (fn [message _send!]
     (let [id        (:com.vadelabs.turbo-ui.explorer.rpc/id message)
           [resolve] (get @pending-requests id)]
       (swap! pending-requests dissoc id)
       (when (fn? resolve) (resolve message))))
   :com.vadelabs.turbo-ui.explorer.rpc/eval-str
   (fn [message send!]
     (let [return
           (fn [msg]
             (send!
               (assoc msg
                 :op            :com.vadelabs.turbo-ui.explorer.rpc/response
                 :com.vadelabs.turbo-ui.explorer.rpc/id (:com.vadelabs.turbo-ui.explorer.rpc/id message))))
           error
           (fn [e]
             (return {:error e :message (.-message e)}))]
       (try
         (let [{:keys [value] :as response} (cljs/eval-string message)]
           (if-not (:await message)
             (return response)
             (-> (.resolve js/Promise value)
               (.then #(return (assoc response :value %)))
               (.catch error))))
         (catch :default e
           (.error js/console e)
           (error e)))))
   :com.vadelabs.turbo-ui.explorer.rpc/invalidate
   (fn [message send!]
     (rt/deref (:atom message))
     (send! {:op :com.vadelabs.turbo-ui.explorer.rpc/response
             :com.vadelabs.turbo-ui.explorer.rpc/id (:com.vadelabs.turbo-ui.explorer.rpc/id message)}))
   :com.vadelabs.turbo-ui.explorer.rpc/close
   (fn [message send!]
     (js/setTimeout
       (fn []
         (state/notify-parent {:type :close})
         (js/window.close))
       100)
     (send! {:op :com.vadelabs.turbo-ui.explorer.rpc/response
             :com.vadelabs.turbo-ui.explorer.rpc/id (:com.vadelabs.turbo-ui.explorer.rpc/id message)}))
   :com.vadelabs.turbo-ui.explorer.rpc/clear
   (fn [message send!]
     (a/do
       (state/dispatch! state/state state/clear)
       (reset! rt/current-values {})
       (send! {:op :com.vadelabs.turbo-ui.explorer.rpc/response
               :com.vadelabs.turbo-ui.explorer.rpc/id (:com.vadelabs.turbo-ui.explorer.rpc/id message)})))
   :com.vadelabs.turbo-ui.explorer.rpc/push-state
   (fn [message send!]
     (state/dispatch! state/state state/history-push {:portal/value (:state message)})
     (send!
       {:op :com.vadelabs.turbo-ui.explorer.rpc/response
        :com.vadelabs.turbo-ui.explorer.rpc/id (:com.vadelabs.turbo-ui.explorer.rpc/id message)}))})

(defn- dispatch [message send!]
  ;; (tap> (assoc message :type :response))
  (when-let [f (get ops (:op message))] (f message send!)))

(defn ^:export ^:no-doc handler [request]
  (write (dispatch (read request) identity)))

(defonce ^:private ws-promise (atom nil))

(defn- get-session []
  (if (exists? js/PORTAL_SESSION)
    js/PORTAL_SESSION
    (subs js/window.location.search 1)))

(defn- get-host []
  (if (exists? js/PORTAL_HOST)
    js/PORTAL_HOST
    (.-hostname js/location)))

(defn- get-proto []
  (if (= (.-protocol js/location) "https:") "wss:" "ws:"))

(defn- get-port []
  (when-not (= (.-port js/location) "")
    (.-port js/location)))

(defn connect
  ([]
   (connect
     {:host     (get-host)
      :port     (get-port)
      :protocol (get-proto)
      :session  (get-session)}))
  ([{:keys [host port protocol session]}]
   (if-let [ws @ws-promise]
     ws
     (reset!
       ws-promise
       (js/Promise.
         (fn [resolve reject]
           (when-let [chan (js/WebSocket.
                             (str protocol "//" host (when port (str ":" port)) "/rpc?" session))]
             (set! (.-onmessage chan) #(dispatch (read (.-data %))
                                         (fn [message]
                                           (send! message))))
             (set! (.-onerror chan)   (fn [e]
                                        (reject e)
                                        (.error js/console (pr-str e))
                                        (doseq [[_ [_ reject]] @pending-requests]
                                          (reject e))))
             (set! (.-onclose chan)   #(reset!  ws-promise nil))
             (set! (.-onopen chan)    #(resolve chan)))))))))

(defn- send! [message]
  (.then (connect) #(.send % (write message))))
