(ns com.vadelabs.turbo-utils.interface
  (:require
   [com.vadelabs.turbo-utils.dom :as dom]
   [com.vadelabs.turbo-utils.object :as obj]
   [com.vadelabs.utils-core.interface :as uc])
  (:import [goog.async Debouncer]))

(defn- parse-browser
  []
  (let [user-agent (-> (dom/get-user-agent) uc/str-lower)
        check-chrome? (fn [] (uc/str-includes? user-agent "chrom"))
        check-firefox? (fn [] (uc/str-includes? user-agent "firefox"))
        check-edge? (fn [] (uc/str-includes? user-agent "edg"))
        check-safari? (fn [] (uc/str-includes? user-agent "safari"))]
    (cond
      (check-edge?)    :edge
      (check-chrome?)  :chrome
      (check-firefox?) :firefox
      (check-safari?)  :safari
      :else            :other)))

(defn- parse-platform
  []
  (let [user-agent     (uc/str-lower (dom/get-user-agent))
        check-windows? (fn [] (uc/str-includes? user-agent "windows"))
        check-linux?   (fn [] (uc/str-includes? user-agent "linux"))
        check-macos?   (fn [] (uc/str-includes? user-agent "mac os"))]
    (cond
      (check-windows?) :windows
      (check-linux?)   :linux
      (check-macos?)   :macos
      :else            :other)))

(defn- parse-target
  [global]
  (if (some? (obj/get global "document"))
    :browser
    :webworker))

(defn is-key?
  [^string key]
  (fn [^js e]
    (= (.-key e) key)))

(defn is-key-ignore-case?
  [^string key]
  (fn [^js e]
    (= (uc/str-upper (.-key e)) (uc/str-upper key))))

(defn ^boolean alt?
  [^js event]
  (.-altKey event))

(defn ^boolean ctrl?
  [^js event]
  (.-ctrlKey event))

(defn ^boolean meta?
  [^js event]
  (.-metaKey event))

(defn ^boolean shift?
  [^js event]
  (.-shiftKey event))

(def browser
  (atom (parse-browser)))
(def platform
  (atom (parse-platform)))

(defn ^boolean check-browser? [candidate]

  (= candidate @browser))

(defn ^boolean check-platform? [candidate]
  (= candidate @platform))

(defn ^boolean mod?
  [^js event]
  (if (check-platform? :macos)
    (meta? event)
    (ctrl? event)))

(def esc?
  (is-key? "Escape"))
(def enter? (is-key? "Enter"))
(def space? (is-key? " "))
(def z? (is-key-ignore-case? "z"))
(def up-arrow? (is-key? "ArrowUp"))
(def down-arrow? (is-key? "ArrowDown"))
(def left-arrow? (is-key? "ArrowLeft"))
(def right-arrow? (is-key? "ArrowRight"))
(def alt-key? (is-key? "Alt"))
(def shift-key? (is-key? "Shift"))
(def ctrl-key? (is-key? "Control"))
(def meta-key? (is-key? "Meta"))
(def comma? (is-key? ","))
(def backspace? (is-key? "Backspace"))
(def home? (is-key? "Home"))
(def tab? (is-key? "Tab"))

(defn editing? [e]
  (.-editing ^js e))

(def stop-propagation
  dom/stop-propagation)

(def prevent-default
  dom/prevent-default)

(defn uuid-string? [x]
  (and (= (count (filter #(= % \-) x)) 4)
    (= (count x) 36)))

(defn parse-id
  "Parse a path parameter from a fulcro route into an ID"
  [x]
  (when x
    (cond
      (uuid? x) x

      (uuid-string? x)
      (uuid x)

      (re-matches #"^\d+$" x)
      (js/parseInt x)

      :else
      (keyword x))))

(defn debounce
  ([f]
   (debounce f 1000))
  ([f interval]
   (let [dbnc (Debouncer. f interval)]
    ;; We use apply here to support functions of various arities
     (fn [& args] (.apply (.-fire dbnc) dbnc (to-array args))))))
