(ns com.vadelabs.turbo-ui.explorer.api
  (:require #?(:clj
               [com.vadelabs.turbo-ui.explorer.runtime.jvm.commands])
            #?(:clj  [com.vadelabs.turbo-ui.explorer.runtime.jvm.launcher :as l]
               :cljs [com.vadelabs.turbo-ui.explorer.runtime.node.launcher :as l]
               :cljr [com.vadelabs.turbo-ui.explorer.runtime.clr.launcher :as l])
            #?(:clj  [com.vadelabs.turbo-ui.explorer.sync :as a]
               :cljs [com.vadelabs.turbo-ui.explorer.async :as a]
               :cljr [com.vadelabs.turbo-ui.explorer.sync :as a])
            #?(:clj  [clojure.java.io :as io]
               :cljs [com.vadelabs.turbo-ui.explorer.resources :as io])
            [clojure.set :as set]
            [com.vadelabs.turbo-ui.explorer.runtime :as rt]
            [com.vadelabs.turbo-ui.explorer.runtime.cson :as cson]))

(defn submit
  "Tap target function.

  Usage:

  ```clojure
  (add-tap #'com.vadelabs.turbo-ui.explorer.api/submit)
  (remove-tap #'com.vadelabs.turbo-ui.explorer.api/submit)
  ```"
  {:added "0.9.0"}
  [value]
  (rt/update-value value)
  nil)

(defn tap
  "Add portal as a `tap>` target."
  {:added         "0.1.0"
   :deprecated    "0.9"
   :superseded-by "submit"}
  []
  (add-tap #'submit)
  nil)

(def ^:private long->short
  {:com.vadelabs.turbo-ui.explorer.colors/theme          :theme
   :com.vadelabs.turbo-ui.explorer.launcher/app          :app
   :com.vadelabs.turbo-ui.explorer.launcher/host         :host
   :com.vadelabs.turbo-ui.explorer.launcher/port         :port
   :com.vadelabs.turbo-ui.explorer.launcher/window-title :window-title})

(defn- rename [options]
  (set/rename-keys options long->short))

(defn set-defaults!
  "Set default options for `open` and `start`.
  Parameters passed directly to either will override defaults."
  {:added    "0.20.0"
   :see-also ["open" "start"]}
  [options]
  (swap! rt/default-options merge (rename options)))

(defn start
  "Start the HTTP server with non-default options. Only use if you need
  control over the HTTP server."
  {:added "0.6.2"
   :see-also ["stop"]}
  [options]
  (l/start (rename options)))

(defn open
  "Open a new inspector window. A previous instance can be passed as
  parameter to make sure it is open."
  {:added "0.1.0"
   :see-also ["close"]}
  ([] (open nil))
  ([portal-or-options]
   (if (:session-id portal-or-options)
     (open portal-or-options nil)
     (open nil portal-or-options)))
  ([portal options]
   (l/open portal (rename options))))

(defn close
  "Close all current inspector windows."
  {:added "0.1.0"
   :see-also ["open"]}
  ([]
   (l/close :all)
   nil)
  ([portal]
   (l/close portal)
   nil))

(defn clear
  "Clear all values."
  {:added "0.1.0"}
  ([]
   (l/clear :all)
   nil)
  ([portal]
   (l/clear portal)
   nil))

(defn register!
  "Register a var with com.vadelabs.turbo-ui.explorer. For now, the var should be a 1 arity fn.

  Example:

  ```clojure
  (register! #'identity)
  ```

  The function name and doc string will show up in the command palette."
  {:added "0.16.0"}
  [var]
  (rt/register! var)
  nil)

(defn- print-err [s]
  #?(:clj  (binding [*out* *err*]
             (print s)
             (flush))
     :cljs (binding [*print-fn* *print-err-fn*]
             (print s)
             (flush))
     :cljr (binding [*out* *err*]
             (print s)
             (flush))))

(defn eval-str
  "Evalute ClojureScript source given as a string in the UI runtime. The parameters:

   - portal: portal instance returned from `com.vadelabs.turbo-ui.explorer.api/open` or `:all`
   - code (string): the ClojureScript source
   - opts (map): evaluation options.
     - `:verbose` optional, return a map containing more info that just the value.
       - Defaults to false.
     - `:await`   - optional, await a promise result. Defaults to `false`."
  {:added "0.19.0"
   :see-also ["open"]}
  ([code]
   (eval-str :all code))
  ([portal code]
   (eval-str portal code nil))
  ([portal code opts]
   (a/let [result (l/eval-str portal (assoc opts :code code))]
     (when-not (:verbose opts)
       (doseq [{:keys [tag val]} (:stdio result)]
         (cond
           (= :out tag) (do (print val) (flush))
           (= :err tag) (print-err val))))
     (cond-> result (not (:verbose opts)) :value))))

(defn sessions
  "Get all current portal sessions."
  {:added "0.27.0"}
  []
  (l/sessions))

(defn url
  "Get url for portal session."
  {:added "0.33.0"}
  [portal]
  (l/url portal))

(def ^:no-doc ^:dynamic *nrepl-init* nil)

(defn repl
  "Start a repl for the given Portal session."
  {:added "0.31.0"}
  [portal]
  (if *nrepl-init*
    (*nrepl-init* portal)
    (throw
      (ex-info
        "Please start nREPL with `com.vadelabs.turbo-ui.explorer.nrepl/wrap-repl` middleware to enable the portal subrepl."
        {:portal-instance    portal
         :missing-middleware 'com.vadelabs.turbo-ui.explorer.nrepl/wrap-repl}))))

(defn- get-docs []
  #_(cson/read
      #?(:clj  (slurp (io/resource "portal/docs.json"))
         :cljs (io/inline "portal/docs.json"))))

(defn docs
  "Open portal docs."
  {:added "035.0"
   :see-also ["open"]}
  ([]
   (docs nil))
  ([options]
   (open (assoc options :window-title "portal-docs" :value (get-docs)))))

(defn inspect
  "Open a new portal window to inspect a particular value."
  {:command true
   :added "0.38.0"
   :see-also ["open"]}
  ([value]
   (inspect value (:options rt/*session*)))
  ([value options]
   (open (assoc options :value value))))

(register! #'inspect)
