(ns com.vadelabs.turbo-core.interface
  "Public API"
  (:require
   [cljs-bean.core :as bean]
   [com.vadelabs.turbo-core.compiler.aot]
   [com.vadelabs.turbo-core.hooks :as hooks]
   [react])
  (:require-macros [com.vadelabs.turbo-core.interface]))

(def ^:dynamic *current-component*)

;; React's top-level API

(def ^:private built-in-static-method-names
  [:childContextTypes :contextTypes :contextType
   :getDerivedStateFromProps :getDerivedStateFromError])

(defn create-ref
  "Creates React's ref type object"
  []
  (react/createRef))

(defn glue-args [^js props]
  (cond-> (.-argv props)
    (.-children props) (assoc :children (.-children props))))

(defn- memo-compare-args [a b]
  (= (glue-args a) (glue-args b)))

(defn memo
  "Takes component `f` and optional comparator function `should-update?`
  that takes previous and next props of the component.
  Returns memoized `f`.

  When `should-update?` is not provided uses default comparator
  that compares props with clojure.core/="
  ([f]
   (memo f memo-compare-args))
  ([^js f should-update?]
   (let [fm (react/memo f should-update?)]
     (when (.-turbo-component? f)
       (set! (.-turbo-component? fm) true))
     fm)))

(defn use-state
  "Takes initial value or a function that computes it and returns a stateful value,
  and a function to update it.

  See: https://reactjs.org/docs/hooks-reference.html#usestate"
  [value]
  (hooks/use-state value))

(defn use-reducer
  "An alternative to `use-state`. Accepts a reducer of type (state, action) => new-state,
  and returns the current state paired with a dispatch method.

  See: https://reactjs.org/docs/hooks-reference.html#usereducer"
  ([f value]
   (hooks/use-reducer f value))
  ([f value init-state]
   (hooks/use-reducer f value init-state)))
(defn use-ref
  "Takes optional initial value and returns React's ref hook wrapped in atom-like type."
  ([]
   (use-ref nil))
  ([value]
   (let [ref (hooks/use-ref nil)]
     (when (nil? (.-current ref))
       (set! (.-current ref)
         (specify! #js {:current value}
           IDeref
           (-deref [this]
             (.-current this))

           IReset
           (-reset! [this v]
             (set! (.-current ^js this) v))

           ISwap
           (-swap!
             ([this f]
              (set! (.-current ^js this) (f (.-current ^js this))))
             ([this f a]
              (set! (.-current ^js this) (f (.-current ^js this) a)))
             ([this f a b]
              (set! (.-current ^js this) (f (.-current ^js this) a b)))
             ([this f a b xs]
              (set! (.-current ^js this) (apply f (.-current ^js this) a b xs)))))))
     (.-current ref))))

(defn create-context
  "Creates React Context with an optional default value"
  ([]
   (react/createContext))
  ([default-value]
   (react/createContext default-value)))

(defn use-context
  "Takes React context and returns its current value"
  [context]
  (hooks/use-context context))

(defn use-deferred-value
  "Accepts a value and returns a new copy of the value that will defer to more urgent updates.
   If the current render is the result of an urgent update, like user input,
   React will return the previous value and then render the new value after the urgent render
   has completed.

   See: https://reactjs.org/docs/hooks-reference.html#usedeferredvalue"
  [v]
  (hooks/use-deferred-value v))

(defn use-transition
  "Returns a stateful value for the pending state of the transition, and a function to start it.

  See: https://reactjs.org/docs/hooks-reference.html#usetransition"
  []
  (hooks/use-transition))

(defn start-transition
  "Marks updates in `f` as transitions
  See: https://reactjs.org/docs/react-api.html#starttransition"
  [f]
  (react/startTransition f))

(defn use-id
  "Returns unique ID that is stable across the server and client, while avoiding hydration mismatches.

  See: https://reactjs.org/docs/hooks-reference.html#useid"
  []
  (hooks/use-id))

(defn use-sync-external-store
  "For reading and subscribing from external data sources in a way that’s compatible
  with concurrent rendering features like selective hydration and time slicing.

  subscribe: function to register a callback that is called whenever the store changes
  get-snapshot: function that returns the current value of the store
  get-server-snapshot: function that returns the snapshot used during server rendering

  See: https://reactjs.org/docs/hooks-reference.html#usesyncexternalstore"
  ([subscribe get-snapshot]
   (hooks/use-sync-external-store subscribe get-snapshot))
  ([subscribe get-snapshot get-server-snapshot]
   (hooks/use-sync-external-store subscribe get-snapshot get-server-snapshot)))

(defn as-react
  "Interop with React components. Takes Turbo component function
  and returns same component wrapped into interop layer."
  [f]
  #(f #js {:argv (bean/bean %)}))

(defn stringify-clojure-primitives [v]
  (cond
    ;; fast direct lookup for a string value
    ;; already stored on the instance of the known type
    (keyword? v) (.-fqn v)
    (uuid? v) (.-uuid v)
    (symbol? v) (.-str v)
    :else v))

(defn jsfy-deps [coll]
  (if (or (js/Array.isArray coll)
        (vector? coll))
    (reduce (fn [arr v]
              (.push arr (stringify-clojure-primitives v))
              arr)
      #js []
      coll)
    coll))

(defn lazy
  "Like React.lazy, but supposed to be used with UIx components"
  [f]
  (let [lazy-component (react/lazy #(.then (f) (fn [component] #js {:default component})))]
    (set! (.-turbo-component? lazy-component) true)
    lazy-component))
