(ns com.vadelabs.turbo-ui.search.interface
  (:require
   ["@searchkit/instantsearch-client$default" :as Client]
   ["algoliasearch" :as algoliasearch]
   ["react-instantsearch-hooks-web" :refer [InstantSearch useHits useRefinementList useSearchBox useSortBy]]
   ["searchkit$default" :as Searchkit]
   [cljs-bean.core :refer [->js ->clj]]
  ;;  [com.fulcrologic.fulcro.components :as fc :refer [defsc]]
  ;;  [com.vadelabs.studio.client.app :as sa]
  ;;  [com.vadelabs.studio.client.mutations :as mutations]
   [com.vadelabs.turbo-core.interface :as tc :refer [defui $]]
   [com.vadelabs.turbo-css.interface :refer [sx]]
   [com.vadelabs.turbo-icons.interface :as ti]
   [com.vadelabs.turbo-ui.button.interface :as button]
   [com.vadelabs.turbo-ui.text.interface :as text]
   [com.vadelabs.utils-core.interface :as uc]))

;;;;;;;;;;;;;;;;;;;;;
;; REFINEMENT LIST ;;
;;;;;;;;;;;;;;;;;;;;;
(defui refinement-list-ui
  "ui for RefinementList"
  [{:keys [items on-refine show-count?]
    :or {show-count? true}}]
  (let [slider-ref (tc/use-ref nil)
        [scroll-end? set-scroll-end!] (tc/use-state false)
        [scroll-start? set-scroll-start!] (tc/use-state true)
        onscroll-handler (fn []
                           ;; check for scroll-start and set scroll-start
                           (if (= (.. slider-ref -current -scrollLeft) 0)
                             (set-scroll-start! true)
                             (set-scroll-start! false))
                           ;; check for scroll-end and set scroll-end
                           (if (<= (Math/floor (- (.. slider-ref -current -scrollWidth)
                                                 (.. slider-ref -current -scrollLeft)))
                                 (.. slider-ref -current -offsetWidth))
                             (set-scroll-end! true)
                             (set-scroll-end! false)))
        slide (fn [num]
                (let [scrlLeft (.. slider-ref -current -scrollLeft)]
                  (set! (.. slider-ref -current -scrollLeft) (+ scrlLeft num));; slide
                  (onscroll-handler)))]
    ($ :div {:class (sx :flex :items-center :gap-0.5)}
      ;;  left scroll btn
      (when (not scroll-start?)
        ($ :button {:class (sx :p-2 :h-full :text-lg :rounded-full
                             [:hover :bg-gray-200])
                    :on-click #(slide (- 50))}
          ($ ti/left-arrow {})))
      ;;  categories
      ($ :ul {:ref slider-ref
              :on-scroll onscroll-handler
              :class (sx :overflow-x-scroll
                       :whitespace-nowrap
                       :flex :gap-2
                       ["::-webkit-scrollbar"
                        :hidden])}
        (mapv
          (fn [{:keys [label value count isRefined] :as item}]
            ;; category
            ($ :li {:class  [(sx :border :border-gray-200 :inline-block
                               :rounded-md)
                             (when isRefined (sx :border :border-gray-800))]
                    :key value}
              ($ :label {:class (sx :cursor-pointer :py-1 :px-2
                                  :inline-flex :gap-1 :items-center)}
                ;; count
                (when isRefined
                  ($ :small {:class  [(sx :bg-gray-200 :rounded :text-xs :p-0.5)]}
                    count))
                ;; category name
                label
                ;; close btn
                (when isRefined
                  ($ :span {:class (sx :text-gray-400)}
                    ($ ti/render {:icon :close})))
                ;; hidden checkbox
                ($ :input {:class (sx :hidden)
                           :value value
                           :checked isRefined
                           :on-change #(on-refine item)
                           :type "checkbox"}))))
          items))
      ;;  right scroll btn
      (when (not scroll-end?)
        ($ :button {:class (sx :p-2 :h-full :text-lg :rounded-full
                             [:hover :bg-gray-200])
                    :on-click #(slide 50)}
          ($ ti/right-arrow {}))))))

(defui refinement-list
  "RefinementList component built using Instantsearch
   Must be used inside of Instantsearch component"
  [{:keys []
    :as props}]
  (let [res (useRefinementList (->js props))
        {:keys [items hasExhaustiveItems createURL refine sendEvent searchForItems isFromSearch canRefine canToggleShowMore isShowingMore toggleShowMore]} (->clj res)
        set-query (fn [new-query]
                    (searchForItems new-query))
        on-refine (fn [item]
                    (refine (:value item))
                    #_(set-query ""))
        ui-props {:items items
                  :canRefine canRefine
                  :on-refine on-refine}]
    ($ refinement-list-ui (merge
                            ui-props
                            props))))

;;;;;;;;;;;;;;;;
;; SEARCH BOX ;;
;;;;;;;;;;;;;;;;
(defui search-box-ui "UI for searchbox component"
  [{:keys [input-ref isSearchStalled on-change on-reset on-submit placeholder value autoFocus translations] :as props}]
  ;; (println "ui: " props)
  (let [handle-submit (fn [e]
                        (.. e (preventDefault))
                        (.. e (stopPropagation))
                        (when on-submit
                          (on-submit e))
                        (when (.. input-ref -current)
                          (.. input-ref -current (blur))))
        handle-reset (fn [e]
                       (.. e (preventDefault))
                       (.. e (stopPropagation))
                       (when on-reset
                         (on-reset e))
                       (when (.. input-ref -current)
                         (.. input-ref -current (focus))))]
    ($ :div {}
      ($ :form {:noValidate ""
                :on-submit handle-submit
                :on-reset handle-reset
                :class (sx :flex :items-center :justify-center
                         :gap-2 :px-3 :py-1.5
                         :w-full :rounded-lg :overflow-hidden
                         :border-2 :border-gray-400
                         [:focus-within :border-gray-700])}
        ($ ti/render  {:icon :search
                       :class (sx :text-gray-600 :text-sm)})
        ($ :label {:htmlFor "search-integrations"
                   :hidden "hidden"}
          "Search for integrations")
        ($ :input {:ref input-ref
                   :id "search-integrations"
                   :class (sx :block :w-full :leading-0
                            :outline-none
                            [:focus :outline-none])
                   :type "search"
                   :value value
                   :on-change on-change
                   :placeholder placeholder
                   :autoFocus autoFocus
                   :autoComplete "off"
                   :autoCorrect "off"
                   :autoCapitalize "off"})
        ($ :button {:type "reset"}
          ($ ti/render  {:icon :close}))))))

(defui search-box
  "SearchBox based on InstantSearch.
   Must be used inside InstantSearch component.
  "
  [{:keys [search-as-you-type translations]
    :as props
    :or {search-as-you-type true}}]
  (let [res (useSearchBox nil)
        {:keys [query refine isSearchStalled]}  (->clj res)
        [input-value set-input-value!] (tc/use-state query)
        input-ref (tc/use-ref nil)
        set-query! (fn [new-query]
                     (set-input-value! new-query)
                     (when search-as-you-type
                       (refine new-query)))
        on-reset (fn []
                   (set-query! "")
                   (when (not search-as-you-type)
                     (refine "")))
        on-submit (fn [e]
                    (when (not search-as-you-type)
                      (refine input-value))
                    (when (props :on-submit)
                      ((props :on-submit) e)))
        on-change (fn [e]
                    (set-query! (.. e -currentTarget -value)))
        ui-props {:input-ref input-ref
                  :isSearchStalled isSearchStalled
                  :value input-value
                  :on-change on-change
                  :on-reset on-reset
                  :on-submit on-submit
                  :translations (uc/deep-merge
                                  {:submit-button-title "Submit the search query"
                                   :reset-button-title "Clear the search query"}
                                  translations)}]
    ;; Track when the InstantSearch query changes to synchronize it with the React state.
    ;; We bypass the state update if the input is focused to avoid concurrent updates when typing.
    (when
      (and (not= query input-value) (not= (.. js/document -activeElement) (.. input-ref -current)))
      (set-input-value! query))

    ($ search-box-ui (uc/deep-merge
                       ui-props
                       props))))

;;;;;;;;;;;;;;;;;;;;
;; SEARCH RESULTS ;;
;;;;;;;;;;;;;;;;;;;;
(defui project-card "Project Card"
  [{:keys [icon name description count]
    :or {count 2}}]
  ($ :div {:class (sx [:w "252px"] #_[:h "252px"] :bg-gray-50 :p-6 :rounded-xl :shadow
                    [:hover [:box-shadow "4px 4px 4px 4px #efefef"]]
                    :flex :flex-col :items-center :text-center :justify-center :gap-3)}
    ($ ti/render {:icon :google-maps
                  :class (sx :text-2xl)})
    ($ text/component {:as :h3} "Google Maps")
    ($ :p {:class (sx :text-xs)} "A web mapping platform and consumer application offered by Google. It offers satellite imagery and so much more.")
    ($ :div {}
      ($ button/component {:class (sx :bg-gray-800)} "Add Integration"))
    ($ :p {:role :button
           :class (sx :text-xs :font-semibold [:hover :underline])} "Read more →")))

;; custom Hits
(defui hits-ui "results UI"
  [{:keys [hits]}]
  ($ :<>
    (vector
      (map-indexed (fn [index value]
                     ($ project-card (assoc value :key index)))
        hits))))

(defui results
  "Hits component based on InstantSearch
   Must be used inside of InstantSearch component"
  [{:keys []}]
  (let [res (useHits)
        {:keys [hits sendEvent]} (->clj res)
        ui-props {:hits hits
                  :sendEvent sendEvent}]
    ($ hits-ui ui-props)))

;;;;;;;;;;;;;
;; SORT BY ;;
;;;;;;;;;;;;;
(defui sortby-ui
  "sortby-ui"
  [{:keys [label items value on-change]}]
  ($ :div {}
    ($ :label {:class (sx :text-gray-400)}
      label
      ($ :select {:on-change #(on-change (.. % -target -value))
                  :value value
                  :class (sx :text-gray-800)}
        (mapv
          (fn [{:keys [value label]}]
            ($ :option {:key value :value value}
              label))
          items)))))

(defui sortby
  "component "
  [{:keys [items label]
    :as props
    :or {label "Sort by: "}}]
  (let [res (useSortBy (->js
                         {:items items}))
        {:keys [options currentRefinement refine]} (->clj res)
        ui-props {:label label
                  :items options
                  :value currentRefinement
                  :on-change refine}]

    ($ sortby-ui  (uc/deep-merge
                    ui-props
                    props))))

;;;;;;;;;;;;;;;;;;;;;;
;; SEARCH CONTAINER ;;
;;;;;;;;;;;;;;;;;;;;;;
(defn client "Configure search client"
  [{:keys [client config]}]
  (cond
    (= client :algolia) (algoliasearch (:app-id config) (:api-key config))
    (= client :searchkit) (-> config
                            ->js
                            (new Searchkit)
                            Client)))

(defui container "search"
  [{:keys [children search-settings] :or {search-settings {:client :algolia
                                                           :index-name "vadelabs_integrations"
                                                           :config {:app-id "1IXQ2W9ZW0"
                                                                    :api-key "1132e0fabfe1291d53ea14f71ddc0320"}}}}]

  (let [search-client (client search-settings)
        index-name (:index-name search-settings)]
    ($ InstantSearch {:indexName index-name
                      :searchClient search-client}
      children)))
