(ns com.vadelabs.turbo-css.tailwind.border
  (:require
   [com.vadelabs.turbo-css.rule :refer [rule]]
   [com.vadelabs.turbo-css.tailwind.tokens :as tokens]
   [com.vadelabs.turbo-css.utils :as utils]
   [com.vadelabs.utils-core.interface :as uc]

   [garden.compiler :refer [render-css]]))

(defn ^:private radius-rules
  [tw-key & css-keys]
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str (uc/namify tw-key) "-" (uc/namify k)))
        (zipmap css-keys (repeat v))))
    {}
    tokens/radius-tokens))

(defn ^:private width-rules
  [tw-key & css-keys]
  (reduce
    (fn [acc k]
      (assoc acc
        (uc/keywordize (uc/str (uc/namify tw-key) "-" (uc/namify k)))
        (zipmap css-keys (repeat (utils/as-unit k :px)))))
    {}
    [0 2 4 8]))

(def ^:private color
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str "border-" (uc/namify k)))
        {:border-color v}))
    {}
    tokens/color-tokens))

(def ^:private style
  {:border-solid	{:border-style "solid"}
   :border-dashed	{:border-style "dashed"}
   :border-dotted	{:border-style "dotted"}
   :border-double	{:border-style "double"}
   :border-hidden	{:border-style "hidden"}
   :border-none	{:border-style "none"}})

(defmethod rule :rounded
  ([_] [[:& {:border-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-radius (tokens/radii x)}]]))

(defmethod rule :rounded-t
  ([_] [[:& {:border-top-left-radius (tokens/radii) :border-top-right-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-top-left-radius (tokens/radii x) :border-top-right-radius (tokens/radii x)}]]))

(defmethod rule :rounded-r
  ([_] [[:& {:border-top-right-radius (tokens/radii)
             :border-bottom-right-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-top-right-radius (tokens/radii x)
               :border-bottom-right-radius (tokens/radii x)}]]))

(defmethod rule :rounded-b
  ([_] [[:& {:border-bottom-right-radius (tokens/radii)
             :border-bottom-left-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-bottom-right-radius (tokens/radii x)
               :border-bottom-left-radius (tokens/radii x)}]]))

(defmethod rule :rounded-l
  ([_] [[:& {:border-top-left-radius (tokens/radii)
             :border-bottom-left-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-top-left-radius (tokens/radii x)
               :border-bottom-left-radius (tokens/radii x)}]]))

(defmethod rule :rounded-tl
  ([_] [[:& {:border-top-left-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-top-left-radius (tokens/radii x)}]]))

(defmethod rule :rounded-tr
  ([_] [[:& {:border-top-right-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-top-right-radius (tokens/radii x)}]]))

(defmethod rule :rounded-br
  ([_] [[:& {:border-bottom-right-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-bottom-right-radius (tokens/radii x)}]]))

(defmethod rule :rounded-bl
  ([_] [[:& {:border-bottom-left-radius (tokens/radii)}]])
  ([_ x] [[:& {:border-bottom-left-radius (tokens/radii x)}]]))

(defmethod rule :rounded-badge
  ([_] [[:& {:border-radius (tokens/radii "1.9rem")}]])
  ([_ x] [[:& {:border-radius (tokens/radii x)}]]))

(defmethod rule :border
  ([_] [[:& {:border-width (utils/as-unit 1 :px)}]])
  ([_ x] [[:& {:border x}]]))

(defmethod rule :border-x
  ([_] (rule :border-x 1))
  ([_ x] [[:& {:border-left-width (utils/as-unit x :px)
               :border-right-width (utils/as-unit x :px)}]]))

(defmethod rule :border-y
  ([_] (rule :border-y 1))
  ([_ x] [[:& {:border-top-width (utils/as-unit x :px)
               :border-bottom-width (utils/as-unit x :px)}]]))

(defmethod rule :border-t
  ([_] (rule :border-t 1))
  ([_ x] [[:& {:border-top-width (utils/as-unit x :px)}]]))

(defmethod rule :border-r
  ([_] (rule :border-r 1))
  ([_ x] [[:& {:border-right-width (utils/as-unit x :px)}]]))

(defmethod rule :border-l
  ([_] (rule :border-l 1))
  ([_ x] [[:& {:border-left-width (utils/as-unit x :px)}]]))

(defmethod rule :border-b
  ([_] (rule :border-b 1))
  ([_ x] [[:& {:border-bottom-width (utils/as-unit x :px)}]]))

;; border
(defmethod  rule :border-top
  ([_ x] [[:& {:border-top x}]]))

(defmethod  rule :border-bottom
  ([_ x] [[:& {:border-bottom x}]]))

(defmethod  rule :border-left
  ([_ x] [[:& {:border-left x}]]))

(defmethod  rule :border-right
  ([_ x] [[:& {:border-right x}]]))

(defmethod  rule :border-top-color
  ([_ x] [[:& {:border-top-color x}]]))

(defmethod  rule :border-bottom-color
  ([_ x] [[:& {:border-bottom-color x}]]))

(defmethod  rule :border-left-color
  ([_ x] [[:& {:border-left-color x}]]))

(defmethod  rule :border-right-color
  ([_ x] [[:& {:border-right-color x}]]))

(defmethod rule :divide
  ([_ x]
   [["&>*+*" {:border-color (utils/with-alpha (tokens/colors x) :--divide-opacity)
              :--divide-opacity 1}]]))

(defmethod rule :divide-opacity
  [_ x] [[:& {:--divide-opacity (utils/as-unit x :percent)}]])

(defmethod rule :divide-x
  ([_] (rule :divide-x 1))
  ([_ x] [["&>*+*" {:--divide-x-reverse 0
                    :border-right-width (uc/str-format "calc(%s * var(--divide-x-reverse))" (render-css (utils/as-unit x :px)))
                    :border-left-width  (uc/str-format "calc(%s * calc(1 - var(--divide-x-reverse)))" (render-css (utils/as-unit x :px)))}]]))

(defmethod rule :divide-y
  ([_] (rule :divide-y 1))
  ([_ x] [["&>*+*" {:--divide-y-reverse  0
                    :border-top-width    (uc/str-format "calc(%s * calc(1 - var(--divide-y-reverse)))" (render-css (utils/as-unit x :px)))
                    :border-bottom-width (uc/str-format "calc(%s * var(--divide-y-reverse))" (render-css (utils/as-unit x :px)))}]]))

(def ^:private outline-width
  {:outline-0	{:outline-width "0px"}
   :outline-1	{:outline-width "1px"}
   :outline-2	{:outline-width "2px"}
   :outline-4	{:outline-width "4px"}
   :outline-8	{:outline-width "8px"}})

(def ^:private outline-color
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str "outline-" (uc/namify k)))
        {:outline-color v}))
    {}
    tokens/color-tokens))

(def ^:private outline-style
  {:outline-none	{:outline "2px solid transparent"
                  :outline-offset "2px"}
   :outline	{:outline-style "solid"}
   :outline-dashed	{:outline-style "dashed"}
   :outline-dotted	{:outline-style "dotted"}
   :outline-double	{:outline-style "double"}
   :outline-hidden	{:outline-style "hidden"}})

(def ^:private outline-offset
  {:outline-offset-0	{:outline-offset "0px"}
   :outline-offset-1	{:outline-offset "1px"}
   :outline-offset-2	{:outline-offset "2px"}
   :outline-offset-4	{:outline-offset "4px"}
   :outline-offset-8	{:outline-offset "8px"}})

(def ^:private ring-width
  {:ring-0	{:box-shadow "var(--tw-ring-inset) 0 0 0 calc(0px + var(--tw-ring-offset-width)) var(--tw-ring-color)"}
   :ring-1	{:box-shadow "var(--tw-ring-inset) 0 0 0 calc(1px + var(--tw-ring-offset-width)) var(--tw-ring-color)"}
   :ring-2	{:box-shadow "var(--tw-ring-inset) 0 0 0 calc(2px + var(--tw-ring-offset-width)) var(--tw-ring-color)"}
   :ring	{:box-shadow "var(--tw-ring-inset) 0 0 0 calc(3px + var(--tw-ring-offset-width)) var(--tw-ring-color)"}
   :ring-4	{:box-shadow "var(--tw-ring-inset) 0 0 0 calc(4px + var(--tw-ring-offset-width)) var(--tw-ring-color)"}
   :ring-8	{:box-shadow "var(--tw-ring-inset) 0 0 0 calc(8px + var(--tw-ring-offset-width)) var(--tw-ring-color)"}
   :ring-inset	{:--tw-ring-inset "inset"}})

(def ^:private ring-color
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str "ring-" (uc/namify k)))
        {:--tw-ring-color v}))
    {}
    tokens/color-tokens))

(def ^:private ring-offset-width
  {:ring-offset-0	{:--tw-ring-offset-width "0px"
                   :box-shadow "0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color), var(--tw-ring-shadow)"}
   :ring-offset-1	{:--tw-ring-offset-width "1px"
                   :box-shadow "0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color), var(--tw-ring-shadow)"}
   :ring-offset-2	{:--tw-ring-offset-width "2px"
                   :box-shadow "0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color), var(--tw-ring-shadow)"}
   :ring-offset-4	{:--tw-ring-offset-width "4px"
                   :box-shadow "0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color), var(--tw-ring-shadow)"}
   :ring-offset-8	{:--tw-ring-offset-width "8px"
                   :box-shadow "0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color), var(--tw-ring-shadow)"}})

(def ^:private ring-offset-color
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str "ring-offset-" (uc/namify k)))
        {:--tw-ring-offset-color v
         :box-shadow "0 0 0 var(--tw-ring-offset-width) var(--tw-ring-offset-color), var(--tw-ring-shadow)"}))
    {}
    tokens/color-tokens))

(def dynamic-rules
  {})

(def static-rules
  (uc/deep-merge
    color
    outline-width
    outline-color
    outline-style
    outline-offset
    (radius-rules :rounded :border-radius)
    (radius-rules :rounded-t :border-top-left-radius :border-top-right-radius)
    (radius-rules :rounded-r :border-top-right-radius :border-bottom-right-radius)
    (radius-rules :rounded-b :border-bottom-right-radius :border-bottom-left-radius)
    (radius-rules :rounded-l :border-top-left-radius :border-bottom-left-radius)
    (radius-rules :rounded-tl :border-top-left-radius)
    (radius-rules :rounded-tr :border-top-right-radius)
    (radius-rules :rounded-br :border-bottom-right-radius)
    (radius-rules :rounded-bl :border-bottom-left-radius)
    ring-width
    ring-color
    ring-offset-width
    ring-offset-color
    style
    (width-rules :border :border-width)
    (width-rules :border-x :border-left-width :border-right-width)
    (width-rules :border-y :border-top-width :border-bottom-width)
    (width-rules :border-t :border-top-width)
    (width-rules :border-r :border-right-width)
    (width-rules :border-l :border-left-width)
    (width-rules :border-b :border-bottom-width)))
