(ns com.vadelabs.turbo-css.tailwind.spacing
  (:require
   [com.vadelabs.turbo-css.rule :refer [rule]]
   [com.vadelabs.turbo-css.utils :as utils]
   [com.vadelabs.utils-core.interface :as uc]
   [garden.compiler :refer [render-css]]))

(def ^:private space-tokens
  {"0" "0px"
   "px" "1px"
   "0.5"  "0.125rem"
   "1" "0.25rem"
   "1.5" "0.375rem"
   "2" "0.5rem"
   "2.5" "0.625rem"
   "3" "0.75rem"
   "3.5" "0.875rem"
   "4" "1rem"
   "5" "1.25rem"
   "6" "1.5rem"
   "7" "1.75rem"
   "8" "2rem"
   "9" "2.25rem"
   "10" "2.5rem"
   "11" "2.75rem"
   "12" "3rem"
   "14" "3.5rem"
   "16" "4rem"
   "20" "5rem"
   "24" "6rem"
   "28" "7rem"
   "32" "8rem"
   "36" "9rem"
   "40" "10rem"
   "44" "11rem"
   "48" "12rem"
   "52" "13rem"
   "56" "14rem"
   "60" "15rem"
   "64" "16rem"
   "72" "18rem"
   "80" "20rem"
   "96" "24rem"
   "auto" "auto"})

(def ^:private padding
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str "p-" k)) {:padding (utils/as-unit v)}
        (uc/keywordize (uc/str "px-" k)) {:padding-right (utils/as-unit v) :padding-left (utils/as-unit v)}
        (uc/keywordize (uc/str "py-" k)) {:padding-top (utils/as-unit v) :padding-bottom (utils/as-unit v)}
        (uc/keywordize (uc/str "pt-" k)) {:padding-top (utils/as-unit v)}
        (uc/keywordize (uc/str "pr-" k)) {:padding-right (utils/as-unit v)}
        (uc/keywordize (uc/str "pb-" k)) {:padding-bottom (utils/as-unit v)}
        (uc/keywordize (uc/str "pl-" k)) {:padding-left (utils/as-unit v)}
        (uc/keywordize (uc/str "-p-" k)) {:padding (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-px-" k)) {:padding-right (utils/as-unit (str "-" v)) :padding-left (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-py-" k)) {:padding-top (utils/as-unit (str "-" v)) :padding-bottom (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-pt-" k)) {:padding-top (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-pr-" k)) {:padding-right (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-pb-" k)) {:padding-bottom (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-pl-" k)) {:padding-left (utils/as-unit (str "-" v))}))
    {}
    space-tokens))

(defmethod rule :pyx [_  y x]
  [[:& {:padding-top    (utils/as-unit y)
        :padding-bottom (utils/as-unit y)
        :padding-left   (utils/as-unit x)
        :padding-right  (utils/as-unit x)}]])

(def ^:private margin
  (reduce-kv
    (fn [acc k v]
      (assoc acc
        (uc/keywordize (uc/str "m-" k)) {:margin (utils/as-unit v)}
        (uc/keywordize (uc/str "mx-" k)) {:margin-right v :margin-left (utils/as-unit v)}
        (uc/keywordize (uc/str "my-" k)) {:margin-top v :margin-bottom (utils/as-unit v)}
        (uc/keywordize (uc/str "mt-" k)) {:margin-top (utils/as-unit v)}
        (uc/keywordize (uc/str "mr-" k)) {:margin-right (utils/as-unit v)}
        (uc/keywordize (uc/str "mb-" k)) {:margin-bottom (utils/as-unit v)}
        (uc/keywordize (uc/str "ml-" k)) {:margin-left (utils/as-unit v)}
        (uc/keywordize (uc/str "-m-" k)) {:margin (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-mx-" k)) {:margin-right (str "-" v) :margin-left (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-my-" k)) {:margin-top (str "-" v) :margin-bottom (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-mt-" k)) {:margin-top (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-mr-" k)) {:margin-right (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-mb-" k)) {:margin-bottom (utils/as-unit (str "-" v))}
        (uc/keywordize (uc/str "-ml-" k)) {:margin-left (utils/as-unit (str "-" v))}))
    {}
    space-tokens))

(defmethod rule :myx [_ y x]
  [[:& {:margin-left   (utils/as-unit x)
        :margin-right  (utils/as-unit x)
        :margin-top    (utils/as-unit y)
        :margin-bottom (utils/as-unit y)}]])

(defmethod rule :space-x
  ([_] (rule :space-x 1))
  ([_ x] [["&>*+*" {:--space-x-reverse 0
                    :margin-right      (uc/str-format "calc(%s * var(--space-x-reverse))" (render-css (utils/as-unit x)))
                    :margin-left       (uc/str-format "calc(%s * calc(1 - var(--space-x-reverse)))" (render-css (utils/as-unit x)))}]]))

(defmethod rule :space-y
  ([_] (rule :space-y 1))
  ([_ x] [["&>*+*" {:--space-y-reverse 0
                    :margin-bottom     (uc/str-format "calc(%s * var(--space-y-reverse))" (render-css (utils/as-unit x)))
                    :margin-top        (uc/str-format "calc(%s * calc(1 - var(--space-y-reverse)))" (render-css (utils/as-unit x)))}]]))

(defmethod rule :space-x-reverse [_] [["&>*+*" {:--space-x-reverse 1}]])
(defmethod rule :space-y-reverse [_] [["&>*+*" {:--space-y-reverse 1}]])

(def dynamic-rules
  {:p (fn [x] {:padding (utils/as-unit x)})
   :px (fn [x] {:padding-left (utils/as-unit x) :padding-right (utils/as-unit x)})
   :py (fn [x] {:padding-top (utils/as-unit x) :padding-bottom (utils/as-unit x)})
   :pt (fn [x] {:padding-top (utils/as-unit x)})
   :pr (fn [x] {:padding-right (utils/as-unit x)})
   :pb (fn [x] {:padding-bottom (utils/as-unit x)})
   :pl (fn [x] {:padding-left (utils/as-unit x)})

   :m (fn [x] {:margin (utils/as-unit x)})
   :mx (fn [x] {:margin-left (utils/as-unit x) :margin-right (utils/as-unit x)})
   :my (fn [x] {:margin-top (utils/as-unit x) :margin-bottom (utils/as-unit x)})
   :mt (fn [x] {:margin-top (utils/as-unit x)})
   :mr (fn [x] {:margin-right (utils/as-unit x)})
   :mb (fn [x] {:margin-bottom (utils/as-unit x)})
   :ml (fn [x] {:margin-left (utils/as-unit x)})})

(def static-rules
  (uc/deep-merge
    padding
    margin))
