(ns com.vadelabs.turbo-css.main
  (:require
   [com.vadelabs.turbo-css.rule :as rule]
   [com.vadelabs.turbo-css.tailwind.animation :as animation]
   [com.vadelabs.turbo-css.tailwind.preflight :as preflight]
   [com.vadelabs.turbo-css.tailwind.tokens :as tokens]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [garden.core]
   [garden.stylesheet])
  #?(:cljs (:require-macros [com.vadelabs.turbo-css.main])))

(defonce styles
  (atom {}))

(defonce media-styles
  (atom {}))

(defonce theme-styles
  (atom {}))

(defonce custom-selector-styles
  (atom {}))

(def ^:private media-rules
  (reduce-kv
    (fn [acc key val]
      (assoc acc key {:screen true
                      :min-width val}))
    (uc/ordered-map)
    tokens/breakpoints))

;; => #ds/ordered-map [[:dark {:screen true, :prefers-color-scheme :dark}] [:sm {:screen true, :min-width "30em"}] [:md {:screen true, :min-width "48em"}] [:lg {:screen true, :min-width "62em"}] [:xl {:screen true, :min-width "80em"}] [:2xl {:screen true, :min-width "96em"}]]

;; => #ds/ordered-map [[:sm {:screen true, :min-width "30em"}] [:md {:screen true, :min-width "48em"}] [:lg {:screen true, :min-width "62em"}] [:xl {:screen true, :min-width "80em"}] [:2xl {:screen true, :min-width "96em"}]]

(defonce media
  (atom {}))

(defn garden-readable
  [media-rules]
  (reduce (fn [acc [f s :as r]]
            (if (string? f)
              (conj acc [(keyword f) (second s)])
              (conj acc r))) [] media-rules))

(defn media-query
  [media-specs class-name rules]
  (garden.stylesheet/at-media
    media-specs
    [[class-name (-> rules
                   rule/join-rules
                   garden-readable)]]))

(defn defmediarules
  [media]
  (doseq [[k v] media]
    (defmethod rule/rule k [_ & rules]
      (fn [class-name]
        (media-query v class-name rules)))))

(defn set-own-mediarules!
  [rules]
  (reset! media {})
  (swap! media merge rules)
  (defmediarules @media)
  @media)

(set-own-mediarules! media-rules)
#_(defn extend-media-rules!
    [rules]
    (swap! merge media rules)
    (defmediarules @media)
    @media)

(defn media-rule?
  [k]
  (when (keyword? k)
    (-> @media
      keys
      (->> (apply hash-set)
        k))))

(defn create-located-classname
  [env]
  (when-let [ns-name (-> env
                       (get-in [:ns :name])
                       uc/namify
                       (ustr/replace #"\.|\-" "_"))]
    (ustr/format ".%s_%s_%s"
      ns-name
      (:line env)
      (:column env))))

(defn create-hashed-classname
  [rules]
  (->> rules
    hash
    (str ".sx")))

(defn create-classname
  [env rules]
  (keyword (or (create-located-classname env)
             (create-hashed-classname rules))))

(defn theme-rule?
  [k]
  (= k :dark))

(defn custom-selector-rule?
  [k]
  (string? k))

(defn divide-rules
  [rules]
  (reduce (fn [acc r]
            (cond
              (keyword? r) (update acc :rules conj r)
              (-> r first media-rule?) (update acc :media-rules conj r)
              (-> r first theme-rule?) (update acc :theme-rules conj r)
              (-> r first custom-selector-rule?) (update acc :custom-selector-rules conj r)
              :else (update acc :rules conj r)))
    {:rules []
     :media-rules []
     :theme-rules []
     :custom-selector-rules []}
    rules))

(defn inject-media-rules
  [class-name garden-obj]
  (swap! media-styles assoc-in
    [class-name
     (-> garden-obj
       :value
       :media-queries)]
    garden-obj))

(defn create-media-rules
  [class-name media-rules]
  (if-not (empty? media-rules)
    (->> media-rules
      (mapv (partial apply rule/rule))
      (mapv (fn [f] (f class-name)))
      (mapv (fn [g] (inject-media-rules class-name g))))
    (swap! media-styles dissoc class-name)))

(defn rules-with-location
  [env rules]
  (with-meta (rule/join-rules rules)
    {:location [(:name (:ns env))
                (:line env)
                (:column env)]}))

(defn create-rules [env rules]
  (when-not (empty? rules)
    (let [class-name (create-classname env rules)]
      (swap! styles dissoc class-name)
      (swap! styles assoc class-name (rules-with-location env rules))
      class-name)))

(defn create-theme-rules
  [class-name theme-rules]
  (doseq [[theme-name & rules] theme-rules]
    (if-not (empty? rules)
      (let [theme-kw (ustr/str "." (uc/namify theme-name))
            crules (rule/join-rules rules)]
        (swap! theme-styles assoc-in [theme-name class-name] [theme-kw [class-name crules]]))
      (swap! theme-styles assoc-in [theme-name class-name] []))))

(defn create-custom-selector-rules
  [class-name attr-rules]
  (when class-name
    (doseq [[custom-selector & rules] attr-rules]
      (if-not (empty? rules)
        (let [custom-selector-selector (ustr/str (uc/namify class-name) custom-selector)
              crules (rule/join-rules rules)]
          (swap! custom-selector-styles assoc-in [custom-selector class-name] [custom-selector-selector crules]))
        (swap! custom-selector-styles assoc-in [custom-selector class-name] [])))))

(defn return-classname
  [classname]
  (->> classname
    str
    (drop 2)
    ustr/join))

(defn c-fn
  [env rs]
  (let [{:keys [media-rules rules theme-rules custom-selector-rules]} (divide-rules rs)
        class-name (or (create-rules env rules)
                     (create-classname env media-rules))
        _ (create-theme-rules class-name theme-rules)
        _ (create-media-rules class-name media-rules)
        _ (create-custom-selector-rules class-name custom-selector-rules)]
    (return-classname class-name)))

(defmacro c? [& rs]
  (let [{:keys [rules media-rules]} (divide-rules rs)
        class-name (if-not (empty? rules)
                     (create-classname &env rules)
                     (create-classname &env media-rules))

        compute-rules (fn [r] (->> r
                                rule/join-rules
                                (into [class-name])
                                garden.core/css
                                boolean))
        compute-media-rules (fn [m] (->> m
                                      (mapv (partial apply rule/rule))
                                      (mapv (fn [f] (f class-name)))
                                      garden.core/css
                                      boolean))]
    (cond
      (and (empty? media-rules) (empty? rules)) true
      (empty? media-rules) (compute-rules rules)
      (empty? rules) (compute-media-rules media-rules)
      :else (and (compute-rules rules)
              (compute-media-rules media-rules)))))

(defn prettify [s]
  (->> s
    (#(ustr/replace % #"\n" ""))
    (#(ustr/replace % #"\s{2,}" " "))
    (reduce (fn [acc v]
              (cond (or (= \{ v)
                      (= \} v)) (conj acc v \newline)
                (= \@ v) (conj acc \newline \newline v)
                :else (conj acc v)))
      [])
    ustr/join))

(defn ^:private sort-media-styles
  [media-style]
  (let [screen-size (get-in media-style [:value :media-queries :min-width])
        rules (get-in media-style [:value :rules])
        classname (-> rules
                    ffirst
                    first)]
    [screen-size classname]))

(defn css-media-styles
  ([]
   (css-media-styles @media-styles))
  ([media-styles]
   (->> media-styles
     vals
     (mapcat vals)
     (sort-by sort-media-styles)
     garden.core/css
     prettify)))

(defn css-theme-styles
  ([]
   (css-theme-styles @theme-styles))
  ([theme-styles]
   (->> theme-styles
     vals
     (mapcat vals)
     garden.core/css
     prettify)))

(defn css-custom-selector-styles
  ([]
   (css-custom-selector-styles))
  ([styles]
   (->> styles
     vals
     (mapcat vals)
     garden.core/css
     prettify)))

(defn css-rules
  ([] (css-rules @styles))
  ([styles]
   (garden.core/css
     (uc/concat-vec
       [preflight/rules]
       animation/keyframes
       (->> styles
         (sort-by (fn [[_ val]] (:location (meta val))))
         (mapv (fn [[k v]] (into [k] v))))))))

(defn get-styles
  []
  (str (css-rules)
    (css-media-styles)
    (css-theme-styles)
    (css-custom-selector-styles)))

(defn compile-styles
  ([styles]
   (str (css-rules styles)))
  ([styles media-styles]
   (str (css-rules styles)
     (css-media-styles media-styles)))
  ([styles media-styles theme-styles]
   (str (css-rules styles)
     (css-media-styles media-styles)
     (css-theme-styles theme-styles)))
  ([styles media-styles theme-styles custom-selector-styles]
   #_(tap> {:s styles
            :ms media-styles
            :ts theme-styles
            :css custom-selector-styles})
   (str (css-rules styles)
     (css-media-styles media-styles)
     (css-theme-styles theme-styles)
     (css-custom-selector-styles custom-selector-styles))))

(comment
  (reset! styles {})
  (reset! theme-styles {})
  (reset! media-styles {})
  (reset! custom-selector-styles {})

  @styles
  @theme-styles
  @custom-selector-styles

  ;; (c-fn nil [[:dark :bg-black]])
  (c-fn nil [["[data-key=\"value\"]" :bg-gray-900]])

  (css-theme-styles @theme-styles)
  (css-custom-selector-styles @custom-selector-styles)

  (c-fn nil [[:text :blue-300] [:smartphone [:text :blue-500]]])
  (c-fn nil [[:smartphone [:text :blue-500] {:font-weight "500"}]])
  ;; (c [:text :blue-300] [:smartphone [:text :blue-500]])
  (c? [:text :blue-300] [:smartphone [:text :blue-500]])
  (c? [:smartphone [:text :blue-500] {:font-weight "500"}]
    [:screen [:text :pink-200] {:font-weight "300"}])
  (c? [:smartphone [:bg :red-500]
       [[:.some-arbitrary-class {:bg :blue-400}]]])
  (c? [:progress-bar [:bg :red-500]] {:font-weight "500"})
  (c? [:progress-bar [:bg :red-500]])
  (c? [:disabled [:hover [:bg :red-500]]])
  (c? [:bg :red-500] [[:.some-arbitrary-class {:bg :blue-400}]])
  (c?
    [:bg :red-500]
    [:hover [[:.some-arbitrary-class {:bg :blue-400}]]])
  (c? [:pseudo ":nth-child(2)" [:hover [:bg :red-500]]])
  (c? [[:& {:color "red"}]
       [:&:target {:color "green"}]])
  (c? {:color "red"})
  (c? [:hover [:placeholder :white] [:mb 1]])
  (c? [:p 1])
  (c? [:placeholder :white])
  (c? [:divide-x 2])

  (def sr-only [:sr-only])
  (c-fn nil sr-only))
