(ns com.vadelabs.utils-core.sql
  (:refer-clojure :exclude [name])
  (:require
   [com.vadelabs.utils-core.data :as udata]
   [com.vadelabs.utils-core.string :as ustr]))

(defn dehyphen
  "Replace _embedded_ hyphens with spaces.
  Hyphens at the start or end of a string should not be touched."
  [s]
  (ustr/replace s #"(\w)-(?=\w)" "$1 "))

(defn literal
  [expr]
  (-> expr
    udata/namify
    ustr/upper
    dehyphen))

(defn to-sql [arg]
  (cond
    (string? arg)
    [arg]
    (sequential? arg)
    arg))

(defn join
  [separator args]
  (let [args (map to-sql args)]
    (into []
      (cons (ustr/join separator
              (remove ustr/blank? (map first args)))
        (apply concat (map rest args))))))

(defn concat-sql [& args]
  (->> (remove nil? args)
    (map to-sql)
    (reduce (fn [stmt [sql & args]]
              (cons (apply str [(first stmt) sql])
                (concat (rest stmt) args)))
      [])))

(def ^:private quote-styles
  {:double-quote ["\"" "\""]
   :single-quote ["'" "'"]
   :backtick ["`" "`"]
   :square-bracket ["[" "]"]})

(def ^:private reserved
  #{"group"
    "cardinality"})

(defn name
  [{:keys [quote-style]} identifier]
  (let [partials (-> identifier udata/namify (ustr/split #"\."))
        [start end] (get quote-styles quote-style)]
    (->> partials
      (map (fn [partial]
             (let [partial (if quote-style
                             partial
                             (ustr/replace partial #"-" "_"))]
               (if (contains? reserved partial)
                 (ustr/format "\"%s\"" partial)
                 (ustr/format "%s%s%s" start partial end)))))
      (ustr/join "."))))
