(ns com.vadelabs.integration.core
  (:require
    [com.vadelabs.utils.data :as u.data]
    [com.vadelabs.utils.id :as u.id]
    [com.wsscode.pathom.viz.ws-connector.core :as pvc]
    [com.wsscode.pathom.viz.ws-connector.pathom3 :as p.connector]
    [com.wsscode.pathom3.connect.indexes :as pci]
    [com.wsscode.pathom3.connect.operation :as pco]))


(defmulti init-aenv!
  (fn [_genv {:adapter/keys [type]}]
    type))


(defmulti halt-aenv!
  (fn [_genv {:adapter/keys [type]}]
    type))


(defmulti introspect-aenv
  (fn [_genv {:adapter/keys [type]}]
    type))


(defmulti run-migration!
  (fn [_genv {:adapter/keys [type]}]
    type))


(defmulti build-pathom-index
  (fn [_genv {:adapter/keys [type]}]
    type))


(defmulti enrich-attributes
  (fn [{:keys [type]} _attributes]
    type))


(defmulti enrich-actions
  (fn [{:keys [type]} _actions]
    type))


(defmulti pathom-op
  (fn [aenv {:action/keys [type]}]
    [(:type aenv) type]))


(defmethod halt-aenv! :default
  [_ _]
  nil)


(defmethod build-pathom-index :default
  [genv {:adapter/keys [type id]}]
  (let [{:keys [actions] :as aenv} (get-in genv [type id])]
    (->> actions
      (mapv (partial pathom-op aenv))
      (pci/register genv))))


(defn enrich-aenv
  [genv {:adapter/keys [qualify-attributes nspace provider type id config]}]
  (cond-> genv
    nspace (assoc-in [type id :migrations-table] (u.data/prefix-keyword "." nspace :migrations-table))
    nspace (assoc-in [type id :nspace] nspace)
    type (assoc-in [type id :type] type)
    type (assoc-in [type id :dialect] (-> type name keyword))
    provider (assoc-in [type id :provider] provider)
    (seq config) (assoc-in [type id :config] (or config {}))
    (boolean? qualify-attributes) (assoc-in [type id :qualify-attributes] qualify-attributes)))


(defn init!
  [{:keys [parser] :as gopts} adapters]
  (let [adapters (if (map? adapters) [adapters] adapters)
        genv (reduce (fn [aenv adapter]
                       (-> aenv
                         (enrich-aenv adapter)
                         (init-aenv! adapter)
                         (introspect-aenv adapter)
                         (run-migration! adapter)
                         (build-pathom-index adapter)))
               gopts
               adapters)]
    (cond-> (assoc genv
              :com.wsscode.pathom3.error/lenient-mode? true
              ::adapters adapters)
      parser (p.connector/connect-env {::pvc/parser-id parser}))))


(defn halt!
  "Halts the datasource adapter if possible"
  [{::keys [adapters] :as genv}]
  (doseq [adapter adapters]
    (halt-aenv! genv adapter)))


(defn aenv
  [genv {:adapter/keys [type id]}]
  (get-in genv [type id]))


(defn alias-attributes-map
  [attrs]
  (->> attrs
    (mapcat (fn [{:attribute/keys [alias-key qualified-alias-key] :as attribute}]
              (cond-> [[alias-key attribute]]
                qualified-alias-key (conj [qualified-alias-key attribute]))))
    (into {})))


(defn original-attributes-map
  [attrs]
  (->> attrs
    (mapcat (fn [{:attribute/keys [original-key qualified-original-key] :as attribute}]
              (cond-> [[original-key attribute]]
                qualified-original-key (conj [qualified-original-key attribute]))))
    (into {})))


(defn integrations->adapters
  [integrations]
  (reduce (fn [acc {:integration/keys [adapter attributes actions]}]
            (conj acc (assoc adapter :adapter/attributes attributes :adapter/actions actions)))
    []
    integrations))


(defn enrich-action
  [aenv action]
  action)


(defn enrich-attribute
  [{:keys [qualify-attributes nspace]}
   {:attribute/keys [id alias-key alias-type cardinality options default description group
                     unique original-ns original-entity original-key original-type optional
                     identity identity-name target target-name type-name]}]
  (let [id (or id (u.id/uuid alias-key))
        qualified-alias-key (if qualify-attributes (u.data/keywordize nspace alias-key) alias-key)
        qualified-target (when target (if qualify-attributes (u.data/keywordize nspace target) target))
        cardinality (or cardinality :one)
        group (or group original-entity :other)]
    (cond-> {:attribute/id id
             :attribute/alias-key alias-key
             :attribute/alias-type alias-type
             :attribute/qualified-alias-key qualified-alias-key
             :attribute/group group
             :attribute/cardinality cardinality
             :attribute/description (or description "")}
      (boolean? optional) (assoc :attribute/optional optional)
      (boolean? unique) (assoc :attribute/unique unique)
      original-ns (assoc :attribute/original-ns original-ns)
      original-entity (assoc :attribute/original-entity original-entity)
      original-key (assoc :attribute/original-key original-key)
      original-type (assoc :attribute/original-type original-type)
      (seq options) (assoc :attribute/options options)
      (or (boolean? default) default) (assoc :attribute/default default)
      identity (assoc :attribute/identity identity
                 :attribute/identity-name identity-name)
      target (assoc :attribute/target target
               :attribute/target-name target-name)
      type-name (assoc :attribute/type-name type-name)
      qualified-target (assoc :attribute/qualified-target qualified-target))))


(defn mutation
  [{:action/keys [op-name params output]} mutation-fn]
  (pco/mutation
    (cond-> {::pco/op-name (symbol op-name)
             ::pco/mutate mutation-fn}
      (seq params) (assoc ::pco/params params)
      (seq output) (assoc ::pco/output output))))


(defn ^:private optional-input
  [input]
  (mapv #(pco/? %) input))


(defn fetch
  [{:action/keys [op-name input params output priority optionals]} resolver-fn]
  (let [input (->> optionals optional-input (into input))]
    (pco/resolver
      (cond-> {::pco/op-name (symbol op-name)
               ::pco/resolve resolver-fn}
        priority (assoc ::pco/priority priority)
        (seq output) (assoc ::pco/output [{op-name output}])
        (seq input) (assoc ::pco/input input)
        (seq params) (assoc ::pco/params params)))))


(defn fetch-one
  [{:action/keys [op-name input params output priority optionals]} resolver-fn]
  (let [input (->> optionals optional-input (into input))]
    (pco/resolver
      (cond-> {::pco/op-name (symbol op-name)
               ::pco/resolve resolver-fn}
        priority (assoc ::pco/priority priority)
        (seq input) (assoc ::pco/input input)
        (seq params) (assoc ::pco/params params)
        (seq output) (assoc ::pco/output output)))))


(defn ->aliased-result
  [aenv ast-nodes result-map]
  (reduce
    (fn [acc {:keys [children dispatch-key type]}]
      (let [result-value (get result-map dispatch-key)]
        (cond
          (= type :join) (assoc acc dispatch-key (if (map? result-value)
                                                   (->aliased-result aenv children result-value)
                                                   (mapv (partial ->aliased-result aenv children) result-value)))
          (= type :prop) (assoc acc dispatch-key result-value))))
    {}
    ast-nodes))


(defn ->aliased-resultset
  [aenv {:keys [children]} results]
  (if (map? results)
    (->aliased-result aenv children results)
    (mapv (partial ->aliased-result aenv children) results)))


(defn ->malli-schema
  [original-ns original-entity attributes]
  (into [:spec {:table (u.data/keywordize original-ns original-entity)}]
    (->> attributes
      (mapv (fn [{:attribute/keys [original-key original-type]}]
              [(u.data/keywordize original-ns original-entity original-key) {} original-type])))))


(defn attributes->malli-schema
  [attributes]
  (->> attributes
    (group-by (juxt :attribute/original-ns :attribute/original-entity))
    (reduce-kv (fn [acc [original-ns original-entity] attributes]
                 (assoc acc (u.data/keywordize original-ns original-entity)
                   (->malli-schema original-ns original-entity attributes)))
      {})))
