(ns com.vadelabs.sql-core.interface
  (:require
   [com.vadelabs.sql-core.database :as database]
   [com.vadelabs.sql-core.hsql]
   [com.vadelabs.sql-core.introspection :as introspection]
   [com.vadelabs.sql-core.migration :as migration]
   [com.vadelabs.sql-core.provider :as provider]
   [com.vadelabs.sql-core.provider.postgres]
   [com.vadelabs.sql-core.spec :as spec]
   [edn-query-language.core :as eql]
   [honey.sql :as hsql]
   [next.jdbc :as jdbc]
   [next.jdbc.result-set :as jdbc.rs]
   [malli.core :as m]
   [com.vadelabs.utils-core.interface :as uc]
   [clojure.walk :as cwalk]))

(def ^:private active-datasource (atom nil))

(defn connect
  [{:keys [config]}]
  (if @active-datasource @active-datasource
    (reset! active-datasource (database/connect config))))

(defn disconnect!
  [{:keys [db-conn]}]
  (database/disconnect! db-conn))

(defn execute-one!
  [datasource query]
  (jdbc/execute-one! datasource query
    {:return-keys true
     :builder-fn jdbc.rs/as-unqualified-kebab-maps}))

(defn execute!
  [datasource query]
  (jdbc/execute! datasource query
    {:return-keys true
     :builder-fn jdbc.rs/as-unqualified-kebab-maps}))

(defn delete!
  [{:keys [datasource] :as aenv} pparams]
  (let [sql-statements (provider/for-delete aenv pparams)]
    (jdbc/with-transaction [tx datasource]
      (mapv (partial execute! tx) sql-statements))))

(defn save!
  [{:keys [datasource] :as aenv} pparams]
  (let [sql-statements (provider/for-save aenv pparams)]
    (jdbc/with-transaction [tx datasource]
      (mapv (partial execute! tx) sql-statements))))

(defn keywordize-keys
  "Recursively transforms all map keys from strings to keywords."
  [{:keys [attributes-map]} m]
  (let [f (fn [[k v]]
            (let [k (if (string? k) (keyword k) k)
                  cardinality (-> attributes-map k :attribute/cardinality)
                  v (if (and (= :one cardinality) (vector? v))
                      (first v)
                      v)]
              [k v]))]
    (cwalk/postwalk (fn [x] (if (map? x) (into {} (map f x)) x)) m)))

(defn query
  [{:keys [datasource] :as aenv} eql-query]
  (->> eql-query
    (provider/for-query aenv)
    (uc/tap->> "SQL QUERY")
    (execute! datasource)
    (keywordize-keys aenv)))

(defn register-entities
  [aenv]
  (reduce-kv
    (fn [acc schema-keyword ast-spec]
      (assoc acc schema-keyword (spec/register! schema-keyword ast-spec)))
    {}
    (-> aenv :next-sql-schema :tables)))

(defn create-schema
  [{:keys [datasource nspace]}]
  (->> {:create-schema [nspace :if-not-exists]}
    hsql/format
    (execute-one! datasource)))

(defn create-migrations-table
  [{:keys [datasource migrations-table]}]
  (->> {:create-table [migrations-table :if-not-exists]
        :with-columns [[:id :uuid [:not nil] [:default [[:gen-random-uuid]]]]
                       [:sql-schema :jsonb [:not nil]]
                       [:hsql-statements :jsonb [:not nil]]
                       [:sql-statements :jsonb [:not nil]]
                       [:created-at [:time-tz] [:not nil] [:default [:now]]]]}
    hsql/format
    (execute-one! datasource)))

(defn get-latest-sql-schema
  [{:keys [datasource migrations-table]}]
  (or (->> {:select [:sql-schema]
            :from migrations-table
            :limit 1
            :order-by [[:created-at :desc]]}
        hsql/format
        (execute-one! datasource))
    {}))

(defn migrate!
  [aenv]
  (migration/execute! aenv))

(defn introspect!
  [aenv]
  (introspection/execute! aenv))
