(ns com.vadelabs.sql-core.eql-back
  (:require
   [com.vadelabs.sql-core.spec :as spec]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [edn-query-language.core :as eql]
   [honey.sql :as hsql]
   [next.jdbc :as jdbc]
   [next.jdbc.result-set :as jdbc.rs]
   [clojure.zip :as cz]))

(defn nspaced-str
  [k]
  (if (qualified-keyword? k)
    (->> k
      ((juxt namespace name))
      (ustr/join "/"))
    k))

(defn reference-keyword?
  [k]
  (-> k namespace (ustr/includes? ".")))

(defn reference-join?
  [{:keys [type key]}]
  (and (= type :join) (qualified-keyword? key) (reference-keyword? key)))

(defn ident-join?
  [{:keys [type key]}]
  (and (= type :join) (uc/is-ident? key)))

(defn root-join?
  [{:keys [type key]}]
  (or
    (and (= type :join) (vector? key) (empty? key))
    (and (= type :join) (qualified-keyword? key) (-> key reference-keyword? not))))

(defn leaf-node?
  [{:keys [type]}]
  (= :prop type))

(defn cte-name
  [{:keys [dispatch-key] :as eql-node}]
  (if (ident-join? eql-node)
    (-> dispatch-key namespace (ustr/split ".") second keyword)
    (-> dispatch-key name keyword)))

(def ^:private reference-cte-name
  :reference)

(defn jsonb-build-object
  [{:keys [children]}]
  (mapcat (fn [{:keys [dispatch-key]}]
            [(nspaced-str dispatch-key) dispatch-key])
    children))

(defn ->jsonb-select-clause
  ([aenv {:keys [dispatch-key] :as eql-node}]
   (->jsonb-select-clause aenv (nspaced-str dispatch-key) eql-node))
  ([_aenv alias eql-node]
   [[[:jsonb_agg [:distinct [:jsonb_strip_nulls (into [:jsonb_build_object] (jsonb-build-object eql-node))]]]
     alias]]))

(defn dispatch-fn
  [_aenv {:keys [type]}]
  type)

(defmulti ->cte-clause
  dispatch-fn)

(defmulti ->cte-from-clause
  dispatch-fn)

(defmethod ->cte-from-clause :root
  [aenv {:keys [children]}]
  (->> children
    (mapcat (partial ->cte-from-clause aenv))
    (into #{})
    (into [])))

(defmethod ->cte-from-clause :join
  [aenv {:keys [dispatch-key query] :as eql-node}]
  (let [from-alias (->> query first namespace keyword)]
    [[{:select (->jsonb-select-clause aenv eql-node)
       :from [[(cte-name eql-node) from-alias]]}
      (nspaced-str dispatch-key)]]))

(defn merge-queries
  [queries]
  (->> queries
    (reduce (fn [acc [_ {:keys [select from where left-join having group-by]}]]
              (cond-> acc
                select (update :select #(into (or % []) select))
                from (update :from #(into (or % []) from))
                where (update :where #(conj (or % []) where))
                left-join (update :left-join #(into (or % []) left-join))
                having (update :having #(conj (or % []) having))
                group-by (update :group-by #(into (or % []) group-by))))
      {})
    (reduce-kv (fn [acc k v]
                 (let [partial (if (#{:having :where} k)
                                 (into [:or] (distinct v))
                                 (into [] (distinct v)))]
                   (assoc acc k partial)))
      {})))

(defmethod ->cte-clause :root
  [aenv {:keys [children]}]
  (->> children
    (mapcat (partial ->cte-clause aenv))
    (group-by first)
    (reduce-kv (fn [acc cte-name queries]
                 (assoc acc cte-name (merge-queries queries)))
      {})
    (into [])))

(defn ->simple-select-clause
  [aenv {:keys [dispatch-key]}]
  [dispatch-key (-> dispatch-key name keyword)])

(defn ->simple-from-clause
  [aenv {:keys [dispatch-key]}]
  [(-> dispatch-key namespace keyword) (-> dispatch-key namespace keyword)])

(defn ->reference-cte-clause
  [{:keys [attributes-map] :as aenv} {:keys [children key] :as eql-node}]
  (tap> {:eql-node eql-node})
  (let [eql-nodes (->> children (filter reference-join?))
        source-entity (->> eql-nodes
                        (map (comp namespace :dispatch-key))
                        (into #{})
                        first)
        [schema source-column] (ustr/split source-entity ".")
        initial-state (cond-> {:select [[(uc/keywordize source-entity :id) :id]]
                               :left-join []
                               :group-by [(uc/keywordize source-entity :id)]
                               :from [[(keyword source-entity) (keyword source-entity)]]}
                        (ident-join? eql-node) (assoc :having (into [:=] key)))]
    [reference-cte-name (reduce (fn [{:keys [select left-join] :as acc} {:keys [dispatch-key] :as eql-node}]
                                  (let [target-key (get-in attributes-map [dispatch-key :attribute/qualified-target])
                                        join-entity (->> dispatch-key
                                                      ((juxt namespace name))
                                                      (ustr/join "-")
                                                      keyword)
                                        join-alias join-entity
                                        target-entity (if target-key
                                                        (->> target-key namespace keyword)
                                                        (->> dispatch-key name uc/singular (uc/prefix-keyword "." schema))) ;; TODO: entity name should be fetched from schema if available
                                        target-alias target-entity
                                        target-column (-> dispatch-key name keyword)]
                                    (assoc acc
                                      :select (into select (->jsonb-select-clause aenv target-column eql-node))
                                      :left-join (into left-join [[join-entity join-alias] [:= (uc/keywordize source-entity :id) (uc/keywordize join-alias source-column)]
                                                                  [target-entity target-alias] [:= (uc/keywordize join-alias target-column) (uc/keywordize target-alias :id)]]))))
                          initial-state
                          eql-nodes)]))

(defn ->complex-select-clause
  [aenv {:keys [dispatch-key type] :as eql-node}]
  (case type
    :prop (->simple-select-clause aenv eql-node)
    :join [(uc/keywordize reference-cte-name (name dispatch-key)) (-> dispatch-key name keyword)]))

(defn ->unreference-cte-clause
  [aenv {:keys [dispatch-key key children] :as eql-node}]
  (let [[schema entity] (-> dispatch-key
                          namespace
                          (ustr/split "."))
        entity (or entity (-> dispatch-key name uc/singular))
        entity (uc/prefix-keyword "." schema entity)]
    [(cte-name eql-node) (cond-> {:select (->> children
                                            (mapv (partial ->complex-select-clause aenv)))
                                  :left-join [[reference-cte-name reference-cte-name]
                                              [:= (uc/keywordize entity :id) (uc/keywordize reference-cte-name :id)]]
                                  :from [[entity entity]]}
                           (ident-join? eql-node) (assoc :where (into [:=] key)))]))

(defn ->simple-cte-clause
  [aenv {:keys [children key] :as eql-node}]
  [[(cte-name eql-node) (cond-> {:select (->> children
                                           (mapv (partial ->simple-select-clause aenv)))
                                 :from (->> children
                                         (map (partial ->simple-from-clause aenv))
                                         (into #{})
                                         (into []))}
                          (vector? key) (assoc :where (into [:=] key)))]])
(defn ->complex-cte-clause
  [aenv eql-node]
  [(->reference-cte-clause aenv eql-node)
   (->unreference-cte-clause aenv eql-node)])

(defmethod ->cte-clause :join
  [aenv {:keys [children] :as eql-node}]
  (if (every? leaf-node? children)
    (->simple-cte-clause aenv eql-node)
    (->complex-cte-clause aenv eql-node)))

(defn ->honey
  [aenv eql-ast]
  (uc/assoc-some {}
    :with (->cte-clause aenv eql-ast)
    :select [:*]
    :from (->cte-from-clause aenv eql-ast)))

(defn ->hsql
  ([eql-query]
   (->hsql {} eql-query))
  ([aenv eql-query]
   (->> eql-query
     eql/query->ast
     (->honey aenv))))

#_(def test-eql-node {:type :root,
                      :children
                      [{:type :join,
                        :dispatch-key :pg.profile/id,
                        :key [:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"],
                        :query
                        [:pg.profile/id
                         :pg.profile/display-name
                         {:pg.profile/active-workspace
                          [:pg.workspace/id {:pg.workspace/owner [:pg.profile/id]}]}],
                        :children
                        [{:type :prop, :dispatch-key :pg.profile/id, :key :pg.profile/id}
                         {:type :prop,
                          :dispatch-key :pg.profile/display-name,
                          :key :pg.profile/display-name}
                         {:type :join,
                          :dispatch-key :pg.profile/active-workspace,
                          :key :pg.profile/active-workspace,
                          :query [:pg.workspace/id {:pg.workspace/owner [:pg.profile/id]}],
                          :children
                          [{:type :prop,
                            :dispatch-key :pg.workspace/id,
                            :key :pg.workspace/id}
                           {:type :join,
                            :dispatch-key :pg.workspace/owner,
                            :key :pg.workspace/owner,
                            :query [:pg.profile/id],
                            :children
                            [{:type :prop,
                              :dispatch-key :pg.profile/id,
                              :key :pg.profile/id}]}]}]}]})

(defn ast-zipper
  "Make a zipper to navigate an AST tree of EDN query language"
  [ast]
  (->> ast
    (cz/zipper
      (fn branch? [x] (and (map? x)
                        (#{:root :join} (:type x))))
      (fn children [x] (:children x))
      (fn make-node [x xs] (assoc x :children (vec xs))))))

(defn zip-iterator [zipper]
  (->> zipper
    (iterate cz/next)
    (take-while (complement cz/end?))))

(defn join? [loc]
  (let [{:keys [children] :as eql-node} (cz/node loc)
        eql-nodes (filter reference-join? children)]
    (or (seq eql-nodes) (reference-join? eql-node))))

(defn ->cte-identifier
  [aenv {:keys [dispatch-key parent-dispatch-key]}]
  (let [entity (-> dispatch-key namespace (ustr/split ".") second)
        attribute-key (-> dispatch-key name keyword)
        parent-key (when parent-dispatch-key (-> parent-dispatch-key name))]
    (if parent-key
      attribute-key
      (keyword entity))))

;; workspace_owner AS (SELECT "pg.workspace" .id AS id,
;;                      JSONB_AGG (DISTINCT JSONB_STRIP_NULLS (JSONB_BUILD_OBJECT ('pg.profile/id', "pg.owner" .id))) AS owner
;;                      FROM pg.workspace AS "pg.workspace"
;;                      LEFT JOIN pg.workspace_owner AS "pg.workspace_owner" ON "pg.workspace" .id = "pg.workspace_owner" .workspace
;;                      LEFT JOIN pg.profile AS "pg.owner" ON "pg.workspace_owner" .owner = "pg.owner" .id
;;                      GROUP BY "pg.workspace" .id),

(defn ->test-cte-statement
  [aenv eql-node]
  (let [cte-identifier (->cte-identifier aenv eql-node)]
    [cte-identifier eql-node]))

(defn enrich-node
  [loc]
  (let [node (cz/node loc)
        parent (cz/up loc)
        root (cz/root loc)]
    (cond-> loc
      ;; parent (cz/edit assoc :parent-dispatch-key (-> parent cz/node :dispatch-key))
      ;; root (cz/edit assoc :root-dispatch-key (-> root  :dispatch-key))
      :always cz/node)))

(defmulti ->cte-statement
  (fn [_aenv {:keys [query-type]}]
    query-type))

(defmethod ->cte-statement :leaf
  [aenv {:keys [dispatch-key children] :as eql-node}]
  (let [entity (namespace dispatch-key)
        [schema source-table] (-> entity (ustr/split "."))]
    [(->cte-identifier aenv eql-node) eql-node]))

(defmethod ->cte-statement :middle
  [aenv {:keys []}])

(defmethod ->cte-statement :root
  [aenv {:keys [dispatch-key] :as eql-node}])

;; workspace_owner AS (SELECT "pg.workspace" .id AS id,
;;                      JSONB_AGG (DISTINCT JSONB_STRIP_NULLS (JSONB_BUILD_OBJECT ('pg.profile/id', "pg.owner" .id))) AS owner
;;                      FROM pg.workspace AS "pg.workspace"
;;                      LEFT JOIN pg.workspace_owner AS "pg.workspace_owner" ON "pg.workspace" .id = "pg.workspace_owner" .workspace
;;                      LEFT JOIN pg.profile AS "pg.owner" ON "pg.workspace_owner" .owner = "pg.owner" .id
;;                      GROUP BY "pg.workspace" .id),

#_(->> test-eql-node
    ast-zipper
    iter-zip
    (filter join?)
    (map enrich-node)
    reverse
    #_(mapv (partial ->test-cte-statement {})))

(comment
  (def ds (jdbc/get-datasource {:jdbcUrl "jdbc:postgresql://localhost:6432/vadedb?user=vadeuser&password=vadepassword"}))

  (defn execute-query
    [datasource query]
    (jdbc/execute! datasource query
      {:return-keys true
       :builder-fn jdbc.rs/as-unqualified-kebab-maps}))

  (def inline-execute (partial execute-query ds))

  (defn inline-format [query] (hsql/format query {:inline true}))

  (->> [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
         [:pg.profile/id
          :pg.profile/display-name
          {:pg.profile/active-workspace [:pg.workspace/id
                                         {:pg.workspace/owner [:pg.profile/id]}]}
          #_{:pg.profile/workspaces [:pg.workspace/id
                                     {:pg.workspace/members [:pg.profile/id]}]}]}]
    eql/query->ast)

  (->> {:with
        [[:reference
          {:select
           [[:pg.profile/id :id]
            [[:jsonb_agg
              [:distinct
               [:jsonb_strip_nulls
                [:jsonb_build_object
                 "pg.workspace/id"
                 :pg.workspace/id
                 "pg.workspace/display-name"
                 :pg.workspace/display-name]]]]
             :active-workspace]],
           :from [[:pg.profile :pg.profile]],
           :left-join
           [[:pg.profile-active-workspace :pg.profile-active-workspace]
            [:= :pg.profile/id :pg.profile-active-workspace/profile]
            [:pg.workspace :pg.active-workspace]
            [:=
             :pg.profile-active-workspace/active-workspace
             :pg.active-workspace/id]],
           :having
           [:or
            [:= :pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]],
           :group-by [:pg.profile/id]}]
         [:profile
          {:select
           [[:pg.profile/id :id]
            [:pg.profile/display-name :display-name]
            [:reference/active-workspace :active-workspace]],
           :from [[:pg.profile :pg.profile]],
           :where
           [:or
            [:= :pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]],
           :left-join
           [[:reference :reference] [:= :pg.profile/id :reference/id]]}]],
        :select [:*],
        :from
        [[{:select
           [[[:jsonb_agg
              [:distinct
               [:jsonb_strip_nulls
                [:jsonb_build_object
                 "pg.profile/id"
                 :pg.profile/id
                 "pg.profile/display-name"
                 :pg.profile/display-name
                 "pg.profile/active-workspace"
                 :pg.profile/active-workspace]]]]
             "pg.profile/id"]],
           :from [[:profile :pg.profile]]}
          "pg.profile/id"]]}
    inline-format)

  (->> [{[:postgres.adapter/id #uuid "0c1ade93-5421-5495-8115-6bc0a65faa60"] [:postgres.adapter/id]}
        {[:postgres.datasource/id #uuid "813a5e3d-a575-51a3-a8ad-5fb75d550931"] [:postgres.datasource/id
                                                                                 {:postgres.datasource/attributes [:postgres.attribute/id]}
                                                                                 {:postgres.datasource/adapter [:postgres.adapter/id
                                                                                                                :postgres.adapter/nspace]}]}
        {[:postgres.datasource/id #uuid "87c99ff7-75e0-5a32-aa02-39024d4f47ab"] [:postgres.datasource/id
                                                                                 {:postgres.datasource/attributes [:postgres.attribute/id]}
                                                                                 {:postgres.datasource/adapter [:postgres.adapter/id
                                                                                                                :postgres.adapter/nspace]}]}]
    ->hsql
    inline-format
    inline-execute)

  (->> [{[:postgres.adapter/id #uuid "0c1ade93-5421-5495-8115-6bc0a65faa60"] [:postgres.adapter/id
                                                                              :postgres.adapter/nspace]}
        {[:postgres.datasource/id #uuid "813a5e3d-a575-51a3-a8ad-5fb75d550931"] [:postgres.datasource/id
                                                                                 :postgres.datasource/display-name]}
        {[:postgres.datasource/id #uuid "87c99ff7-75e0-5a32-aa02-39024d4f47ab"] [:postgres.datasource/id
                                                                                 :postgres.datasource/display-name]}]
    ->hsql
    inline-format
    inline-execute)

  (->> [{[:postgres.datasource/id #uuid "813a5e3d-a575-51a3-a8ad-5fb75d550931"] [:postgres.datasource/id
                                                                                 :postgres.datasource/display-name
                                                                                 {:postgres.datasource/adapter [:postgres.adapter/id
                                                                                                                :postgres.adapter/nspace]}
                                                                                 {:postgres.datasource/attributes [:postgres.attribute/id
                                                                                                                   :postgres.attribute/local-key]}]}]
    ->hsql
    inline-format
    inline-execute)

  (->> [{[:postgres.datasource/id #uuid "813a5e3d-a575-51a3-a8ad-5fb75d550931"] [:postgres.datasource/id
                                                                                 :postgres.datasource/display-name]}]
    ->hsql
    #_inline-format
    #_inline-execute)
  ;; => {:with
  ;;     [[:datasource
  ;;       {:select [[:postgres.datasource/id :id] [:postgres.datasource/display-name :display-name]],
  ;;        :from [[:postgres.datasource :postgres.datasource]],
  ;;        :where [:or [:= :postgres.datasource/id #uuid "813a5e3d-a575-51a3-a8ad-5fb75d550931"]]}]],
  ;;     :select [:*],
  ;;     :from
  ;;     [[{:select [[[:jsonb_agg [:distinct #]] "postgres.datasource/id"]], :from [[:datasource :postgres.datasource]]}
  ;;       "postgres.datasource/id"]]}

  (->> [{:postgres/datasources [:postgres.datasource/id
                                :postgres.datasource/display-name
                                {:postgres.datasource/adapter [:postgres.adapter/id
                                                               :postgres.adapter/nspace]}
                                {:postgres.datasource/attributes [:postgres.attribute/id
                                                                  :postgres.attribute/local-key]}]}]
    ->hsql
    inline-format
    inline-execute)

;; SIMPLE MANY
  (->> [{:postgres/adapters [:postgres.adapter/id
                             :postgres.adapter/nspace]}
        {:postgres/datasources [:postgres.datasource/id
                                :postgres.datasource/display-name]}]
    ->hsql
    inline-format
    inline-execute)

  {:with [[:cte-name :as]]}

  :rcf)

;; WITH
;; tworef AS (SELECT "pg.workspace" .id AS id,
;;             JSONB_AGG (DISTINCT JSONB_STRIP_NULLS (JSONB_BUILD_OBJECT ('pg.profile/id', "pg.owner" .id))) AS owner
;;             FROM pg.workspace AS "pg.workspace"
;;             LEFT JOIN pg.workspace_owner AS "pg.workspace_owner" ON "pg.workspace" .id = "pg.workspace_owner" .workspace
;;             LEFT JOIN pg.profile AS "pg.owner" ON "pg.workspace_owner" .owner = "pg.owner" .id
;;             GROUP BY "pg.workspace" .id),
;; reference AS (SELECT
;;                "pg.profile" .id AS id,
;;                JSONB_AGG (DISTINCT JSONB_STRIP_NULLS (JSONB_BUILD_OBJECT ('pg.workspace/id', "pg.active_workspace" .id, 'pg.workspace/owner', tworef.owner))) AS active_workspace
;;                FROM pg.profile AS "pg.profile"
;;                LEFT JOIN pg.profile_active_workspace AS "pg.profile_active_workspace" ON "pg.profile" .id = "pg.profile_active_workspace" .profile
;;                LEFT JOIN pg.workspace AS "pg.active_workspace" ON "pg.profile_active_workspace" .active_workspace = "pg.active_workspace" .id
;;                LEFT JOIN tworef AS tworef ON "pg.active_workspace" .id = tworef.id
;;                GROUP BY "pg.profile" .id
;;                HAVING ("pg.profile" .id = 'bead2ebf-3299-53aa-a07d-87bc386608ab')),
;; profile AS (SELECT "pg.profile" .id AS id,
;;              "pg.profile" .display_name AS display_name,
;;              reference.active_workspace AS active_workspace
;;              FROM pg.profile AS "pg.profile"
;;              LEFT JOIN reference AS reference ON "pg.profile" .id = reference.id
;;              WHERE ("pg.profile" .id = 'bead2ebf-3299-53aa-a07d-87bc386608ab'))
;; SELECT * FROM reference
;; -- (SELECT JSONB_AGG (DISTINCT JSONB_STRIP_NULLS (JSONB_BUILD_OBJECT ('pg.profile/id', "pg.profile" .id, 'pg.profile/display-name', "pg.profile" .display_name, 'pg.profile/active-workspace', "pg.profile" .active_workspace))) AS "pg.profile/id" FROM profile AS "pg.profile") AS "pg.profile/id"
