(ns com.vadelabs.sql-core.eql
  (:require
   [clojure.zip :as cz]
   [com.vadelabs.sql-core.spec :as spec]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [edn-query-language.core :as eql]
   [honey.sql :as hsql]
   [next.jdbc :as jdbc]
   [next.jdbc.result-set :as jdbc.rs]))

(defn ast-zipper
  "Make a zipper to navigate an AST tree of EDN query language"
  [ast]
  (->> ast
    (cz/zipper
      (fn branch? [x] (and (map? x)
                        (#{:root :join} (:type x))))
      (fn children [x] (:children x))
      (fn make-node [x xs] (assoc x :children (vec xs))))))

(defn zip-iterator [zipper]
  (->> zipper
    (iterate cz/next)
    (take-while (complement cz/end?))))

(defn ident-join?
  [{:keys [type key]}]
  (and (= type :join) (uc/is-ident? key)))

(defn reference-keyword?
  [k]
  (-> k namespace (ustr/includes? ".")))

(defn reference-join?
  [{:keys [type key]}]
  (and (= type :join) (qualified-keyword? key) (reference-keyword? key)))

(defn root-join?
  [{:keys [type key]}]
  (or
    (and (= type :join) (vector? key) (empty? key))
    (and (= type :join) (qualified-keyword? key) (-> key reference-keyword? not))))

(defn leaf-node?
  [{:keys [type]}]
  (= :prop type))

(defn leaf-join?
  [{:keys [children]}]
  (every? leaf-node? children))

(defn join?
  [{:keys [type]}]
  (= :join type))

(defn nested-join?
  [eql-node]
  (-> eql-node leaf-join? not))

(defn ->cte-identifier
  [{:keys [dispatch-key] :as ast-node}]
  (if (ident-join? ast-node)
    (-> dispatch-key namespace (ustr/split ".") second keyword)
    (-> dispatch-key name keyword)))

(defn enrich-leaf-join
  [loc]
  (let [{:keys [dispatch-key children] :as node} (cz/node loc)
        source-entity (-> dispatch-key namespace keyword)
        source-alias source-entity
        source-key :id
        bridge-target-key (-> dispatch-key name keyword)
        bridge-source-key (-> dispatch-key namespace (ustr/split ".") second keyword)
        bridge-entity (uc/prefix-keyword "-" source-entity bridge-target-key)
        bridge-alias bridge-entity
        target-entity (-> children first :dispatch-key namespace keyword)
        target-alias target-entity
        target-key :id]
    (-> loc
      (cz/edit assoc
        :cte-identifier (->cte-identifier node)
        :source-entity source-entity
        :source-alias source-alias
        :source-key source-key
        :bridge-entity bridge-entity
        :bridge-alias bridge-alias
        :bridge-source-key bridge-source-key
        :bridge-target-key bridge-target-key
        :target-entity target-entity
        :target-alias target-alias
        :target-key target-key
        :query-type :leaf-join)
      cz/node)))

(defn enrich-root-join
  [loc]
  (let [{:keys [dispatch-key children] :as node} (cz/node loc)
        source-entity (->> dispatch-key
                        ((juxt namespace (comp uc/singular name)))
                        (apply uc/prefix-keyword "."))
        source-alias source-entity
        source-key :id
        bridge-target-key (-> dispatch-key name keyword)
        bridge-source-key (-> dispatch-key namespace (ustr/split ".") second keyword)
        bridge-entity (uc/prefix-keyword "-" source-entity bridge-target-key)
        bridge-alias bridge-entity
        target-entity  (-> children first :dispatch-key namespace keyword)
        target-alias target-entity
        target-key :id]
    (-> loc
      (cz/edit assoc
        :cte-identifier (->cte-identifier node)
        :source-entity source-entity
        :source-alias source-alias
        :source-key source-key
        :bridge-entity bridge-entity
        :bridge-alias bridge-alias
        :bridge-source-key bridge-source-key
        :bridge-target-key bridge-target-key
        :target-entity target-entity
        :target-alias target-alias
        :target-key target-key
        :query-type :root-join)
      cz/node)))

(defn enrich-nested-join
  [loc]
  (let [{:keys [dispatch-key children] :as node} (cz/node loc)
        source-entity (-> dispatch-key namespace keyword)
        source-alias source-entity
        source-key :id
        bridge-target-key (-> dispatch-key name keyword)
        bridge-source-key (-> dispatch-key namespace (ustr/split ".") second keyword)
        bridge-entity (uc/prefix-keyword "-" source-entity bridge-target-key)
        bridge-alias bridge-entity
        target-entity  (-> children first :dispatch-key namespace keyword)
        target-alias target-entity
        target-key :id]
    #_(tap> {:node node
             :bridge-entity bridge-entity
             :bridge-alias bridge-alias
             :target-entity target-entity
             :target-alias target-alias})
    (-> loc
      (cz/edit  assoc
        :cte-identifier (->cte-identifier node)
        :source-entity source-entity
        :source-alias source-alias
        :source-key source-key
        :bridge-entity bridge-entity
        :bridge-alias bridge-alias
        :bridge-source-key bridge-source-key
        :bridge-target-key bridge-target-key
        :target-entity target-entity
        :target-alias target-alias
        :target-key target-key
        :query-type :nested-join)
      cz/node)))

(defn enrich-ident-join
  [loc]
  (let [{:keys [dispatch-key] :as node} (cz/node loc)
        source-entity (-> dispatch-key namespace keyword)
        source-alias source-entity
        source-key :id
        bridge-target-key (-> dispatch-key name keyword)
        bridge-source-key (-> dispatch-key namespace (ustr/split ".") second keyword)
        bridge-entity (uc/prefix-keyword "-" source-entity bridge-target-key)
        bridge-alias bridge-entity
        target-entity :hello
        target-alias target-entity
        target-key :id]
    (-> loc
      (cz/edit  assoc
        :cte-identifier (->cte-identifier node)
        :source-entity source-entity
        :source-alias source-alias
        :source-key source-key
        :bridge-entity bridge-entity
        :bridge-alias bridge-alias
        :bridge-source-key bridge-source-key
        :bridge-target-key bridge-target-key
        :target-entity target-entity
        :target-alias target-alias
        :target-key target-key
        :query-type :ident-join)
      cz/node)))

(defn enrich-ast-node
  ([aenv loc]
   (let [node (cz/node loc)]
     #_(tap> {:node node
              :left-join? (leaf-join? node)
              :nested-join? (nested-join? node)
              :root-join? (root-join? node)
              :ident-join? (ident-join? node)})
     (cond
       (leaf-join? node) (enrich-leaf-join loc)
       (and (nested-join? node) (root-join? node)) (enrich-root-join loc)
       (and (nested-join? node) (not (ident-join? node))) (enrich-nested-join loc)
       (ident-join? node) (enrich-ident-join loc))))
  ([loc]
   (enrich-ast-node {} loc)))

(defn nspaced-str
  [k]
  (if (qualified-keyword? k)
    (->> k
      ((juxt namespace name))
      (ustr/join "/"))
    k))

(defn join-node? [loc]
  (let [{:keys [children] :as eql-node} (cz/node loc)
        eql-nodes (filter reference-join? children)]
    (or (seq eql-nodes) (reference-join? eql-node))))

(defn jsonb-build-object
  [{:keys [children]}]
  (mapcat (fn [{:keys [dispatch-key] :as eql-node}]
            (let [column-alias (if (join? eql-node)
                                 (uc/keywordize (->cte-identifier eql-node) (-> dispatch-key name keyword))
                                 dispatch-key)]
              [(nspaced-str dispatch-key) column-alias]))
    children))

(defn ->jsonb-select-clause
  ([{:keys [dispatch-key] :as eql-node}]
   (->jsonb-select-clause (nspaced-str dispatch-key) eql-node))
  ([alias eql-node]
   [[[:jsonb_agg [:distinct [:jsonb_strip_nulls (into [:jsonb_build_object] (jsonb-build-object eql-node))]]]
     alias]]))

(defn inline-format
  [query]
  (hsql/format query {:inline true}))

(defn ->leaf-join-cte-clause
  [{:keys [source-entity source-alias source-key bridge-entity bridge-alias
           bridge-source-key bridge-target-key target-entity target-alias target-key]
    :as eql-node}]
  (let [qualified-source-key (uc/keywordize source-alias source-key)]
    {:select (into [[qualified-source-key source-key]]
               (->jsonb-select-clause bridge-target-key eql-node))
     :from [[source-entity source-alias]]
     :left-join [[bridge-entity bridge-alias]
                 [:= (uc/keywordize bridge-alias bridge-source-key) (uc/keywordize source-alias source-key)]

                 [target-entity target-alias]
                 [:= (uc/keywordize bridge-alias bridge-target-key) (uc/keywordize target-alias target-key)]]
     :group-by [qualified-source-key]}))

(defn ->nested-join-cte-clause
  [{:keys [source-entity source-alias source-key bridge-entity bridge-alias bridge-source-key
           bridge-target-key target-entity target-alias target-key children]
    :as eql-node}]
  (let [qualified-source-key (uc/keywordize source-alias source-key)
        select-clause (->jsonb-select-clause bridge-target-key eql-node)
        left-joins (->> children
                     (filter reference-join?)
                     (mapcat (fn [eql-node]
                               (let [cte-identifier (->cte-identifier eql-node)]
                                 [[cte-identifier cte-identifier]
                                  [:= (uc/keywordize cte-identifier :id) (uc/keywordize target-entity target-key)]]))))]
    {:select (into [[qualified-source-key source-key]] select-clause)
     :from [[source-entity source-alias]]
     :left-join (into [[bridge-entity bridge-alias]
                       [:= (uc/keywordize bridge-alias bridge-source-key) (uc/keywordize source-alias source-key)]

                       [target-entity target-alias]
                       [:= (uc/keywordize bridge-alias bridge-target-key) (uc/keywordize target-alias target-key)]]
                  left-joins)
     :group-by [qualified-source-key]}))

(defn ->ident-join-cte-clause
  [{:keys [source-entity source-alias source-key children]}]
  {:select (->> children
             (mapv (fn [{:keys [type dispatch-key] :as eql-node}]
                     (if (= :prop type)
                       [dispatch-key (nspaced-str dispatch-key)]
                       [(uc/keywordize (->cte-identifier eql-node) (-> dispatch-key name keyword)) (nspaced-str dispatch-key)]))))
   :from [[source-entity source-alias]]
   :left-join (->> children
                (filter reference-join?)
                (mapcat (fn [eql-node]
                          (let [cte-identifier (->cte-identifier eql-node)]
                            [[cte-identifier cte-identifier]
                             [:= (uc/keywordize cte-identifier :id) (uc/keywordize source-entity source-key)]])))
                (into []))})

(defmulti ->cte-clause :query-type)

(defmethod ->cte-clause :ident-join
  [ast-node]
  [(->cte-identifier ast-node) (->ident-join-cte-clause ast-node)])

(defmethod ->cte-clause :root-join
  [ast-node]
  [(->cte-identifier ast-node) (->ident-join-cte-clause ast-node)])

(defmethod ->cte-clause :nested-join
  [ast-node]
  [(->cte-identifier ast-node) (->nested-join-cte-clause ast-node)])

(defmethod ->cte-clause :leaf-join
  [ast-node]
  [(->cte-identifier ast-node) (->leaf-join-cte-clause ast-node)])

(defn distinct-by
  "Returns a stateful transducer that removes elements by calling f on each step as a uniqueness key.
   Returns a lazy sequence when provided with a collection."
  ([f]
   (fn [rf]
     (let [seen (volatile! #{})]
       (fn
         ([] (rf))
         ([result] (rf result))
         ([result input]
          (let [v (f input)]
            (if (contains? @seen v)
              result
              (do (vswap! seen conj v)
                (rf result input)))))))))
  ([f xs]
   (sequence (distinct-by f) xs)))

(defn dispatch-fn
  [_aenv {:keys [type]}]
  type)

(defmulti ->cte-from-clause
  dispatch-fn)

(defmethod ->cte-from-clause :root
  [aenv {:keys [children]}]
  (->> children
    (mapcat (partial ->cte-from-clause aenv))
    (into #{})
    (into [])))

(defmethod ->cte-from-clause :join
  [aenv {:keys [dispatch-key query] :as eql-node}]
  (let [from-alias (->> query first namespace keyword)]
    [[{:select (->jsonb-select-clause aenv eql-node)
       :from [[(-> dispatch-key namespace keyword) from-alias]]}
      (nspaced-str dispatch-key)]]))

(defn ->simple-select-clause
  [{:keys [dispatch-key]}]
  [dispatch-key (nspaced-str dispatch-key)])

(defmulti ->select-clause
  (fn [_aenv {:keys [type]}]
    type))

(defmulti ->from-clause
  (fn [_aenv {:keys [type]}]
    type))

(defn entity-keyword
  [dispatch-key]
  (let [[schema entity] ((juxt namespace name) dispatch-key)]
    (uc/prefix-keyword "." schema (uc/singular entity))))

(defmethod ->select-clause :prop
  [aenv {:keys [dispatch-key]}]
  [dispatch-key (nspaced-str dispatch-key)])

(defmethod ->select-clause :join
  [aenv {:keys [children]}]
  (mapv (partial ->select-clause aenv) children))

(defmethod  ->select-clause :root
  [aenv {:keys [children] :as ast-node}]
  (if (leaf-join? ast-node)
    [:*]
    (->> children
      (mapcat (partial ->select-clause aenv))
      (into []))))

(defmethod ->from-clause :prop
  [aenv {:keys [dispatch-key]}]
  (entity-keyword dispatch-key))

(defmethod ->from-clause :join
  [aenv {:keys [dispatch-key] :as ast-node}]
  (if (root-join? ast-node)
    (uc/prefix-keyword "." (-> dispatch-key namespace keyword) (-> dispatch-key name uc/singular keyword))
    (-> dispatch-key namespace keyword)))

(defmethod  ->from-clause :root
  [aenv {:keys [children]}]
  (->> children
    (map (partial ->from-clause aenv))
    (into #{})
    first))

(defn ->hsql
  ([eql-query]
   (->hsql {} eql-query))
  ([aenv eql-query]
   (let [ast (if (map? eql-query) eql-query (eql/query->ast eql-query))
         _ (tap> {:hsql-ast ast})
         cte-clauses (->> ast
                       ast-zipper
                       zip-iterator
                       (filter join-node?)
                       (map (partial enrich-ast-node aenv))
                       reverse
                       (distinct-by :cte-identifier)
                       (mapv ->cte-clause))
         from-entity (if (seq cte-clauses)
                       (-> cte-clauses last first)
                       (->from-clause aenv ast))
         select-clause (if (seq cte-clauses)
                         [:*]
                         (->select-clause aenv ast))]
     (cond-> {:select select-clause
              :from [[from-entity from-entity]]}
       (seq cte-clauses) (assoc :with cte-clauses)))))

#_(->> [{:pg/profiles [:pg.profile/display-name]}]
    eql/query->ast
    (->select-clause {}))

#_(->> [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
         [:pg.profile/id
          :pg.profile/display-name
          {:pg.profile/active-workspace [:pg.workspace/id]}]}]
    ->hsql
    inline-format)

#_(->> [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
         [:pg.profile/id :pg.profile/display-name]}]
    ->hsql
    inline-format)

#_(->> {:type :join,
        :dispatch-key :pg.profile/id,
        :key [:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"],
        :query
        [:pg.profile/id
         :pg.profile/display-name
         {:pg.profile/active-workspace [:pg.workspace/id]}],
        :children
        [{:type :prop, :dispatch-key :pg.profile/id, :key :pg.profile/id}
         {:type :prop,
          :dispatch-key :pg.profile/display-name,
          :key :pg.profile/display-name}
         {:type :join,
          :dispatch-key :pg.profile/active-workspace,
          :key :pg.profile/active-workspace,
          :query [:pg.workspace/id],
          :children
          [{:type :prop,
            :dispatch-key :pg.workspace/id,
            :key :pg.workspace/id}]}]}
    ast-zipper
    zip-iterator
    (filter join-node?)
    (map (partial enrich-ast-node {})))

(comment

  (def ds (jdbc/get-datasource
            {:jdbcUrl "jdbc:postgres://localhost:6432/vadedb?user=vadeuser&password=vadepassword"}))

  (defn execute-query
    [datasource query]
    (jdbc/execute! datasource query
      {:return-keys true
       :builder-fn jdbc.rs/as-unqualified-kebab-maps}))

  (def inline-execute (partial execute-query ds))

  (->> [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
         [:pg.profile/id
          :pg.profile/display-name]}]
    eql/query->ast
    ast-zipper
    zip-iterator
    (filter (complement join-node?))
    (mapv cz/node))

  (->> [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
         [:pg.profile/id
          :pg.profile/display-name]}]
    ->hsql
    inline-format
    inline-execute)

  (->> [{:pg/profiles [:pg.profile/id
                       :pg.profile/display-name
                       {:pg.profile/active-workspace [:pg.workspace/id
                                                      {:pg.workspace/owner [:pg.profile/id]}
                                                      {:pg.workspace/members [:pg.profile/id]}]}
                       {:pg.profile/workspaces [:pg.workspace/id
                                                {:pg.workspace/owner [:pg.profile/id]}
                                                {:pg.workspace/members [:pg.profile/id]}]}]}]
    ->hsql
    inline-format
    inline-execute)

  (->> [{[:pg.profile/id #uuid "bead2ebf-3299-53aa-a07d-87bc386608ab"]
         [:pg.profile/id
          :pg.profile/display-name
          #_{:pg.profile/active-workspace [:pg.workspace/id
                                           {:pg.workspace/owner [:pg.profile/id]}
                                           {:pg.workspace/members [:pg.profile/id]}]}
          #_{:pg.profile/workspaces [:pg.workspace/id
                                     {:pg.workspace/owner [:pg.profile/id]}
                                     {:pg.workspace/members [:pg.profile/id]}]}]}]
    ->hsql
    inline-format
    inline-execute)

  (->> [{:pg/profiles [:pg.profile/id
                       :pg.profile/display-name
                       {:pg.profile/active-workspace [:pg.workspace/id
                                                      {:pg.workspace/owner [:pg.profile/id]}
                                                      {:pg.workspace/members [:pg.profile/id]}]}
                       {:pg.profile/workspaces [:pg.workspace/id
                                                {:pg.workspace/owner [:pg.profile/id]}
                                                {:pg.workspace/members [:pg.profile/id]}]}]}]
    eql/query->ast
    ast-zipper
    zip-iterator
    (filter join-node?)
    (map enrich-ast-node)
    reverse)
  ;; => ["SELECT * FROM pg.profile AS "pg.profile""]

  (->> [#_{[:pg.adapter/id #uuid "0c1ade93-5421-5495-8115-6bc0a65faa60"] [:pg.adapter/id]}
        #_{[:pg.datasource/id #uuid "813a5e3d-a575-51a3-a8ad-5fb75d550931"] [:pg.datasource/id
                                                                             {:pg.datasource/attributes [:pg.attribute/id]}
                                                                             {:pg.datasource/adapter [:pg.adapter/id
                                                                                                      :pg.adapter/nspace]}]}
        {[:postgres.datasource/id #uuid "87c99ff7-75e0-5a32-aa02-39024d4f47ab"]
         [:postgres.datasource/id
          {:postgres.datasource/attributes [:postgres.attribute/id]}
          {:postgres.datasource/adapter [:postgres.adapter/id
                                         :postgres.adapter/nspace]}]}]
    ->hsql
    inline-format
    inline-execute)

  :rcf)
