(ns com.vadelabs.sql-core.provider.postgres
  (:require

   [com.vadelabs.sql-core.types :as types]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [honey.sql :as hsql]
   [malli.core :as m]
   [next.jdbc :as jdbc]
   [next.jdbc.date-time :as jdbc.dt]
   [next.jdbc.prepare :as jdbc.prep]
   [next.jdbc.result-set :as jdbc.rs]
   [next.jdbc.types :refer [as-other]])
  (:import
   [javax.sql DataSource]
   [java.sql PreparedStatement Time Array]
   [org.postgresql.util PGobject PGInterval]
   [com.vadelabs.sql_core.entity Entity]
   [clojure.lang IPersistentMap IPersistentVector]
   [java.time Duration]))

;; =============================================================================
;; Postgres Array helpers
;; =============================================================================

(extend-protocol jdbc.rs/ReadableColumn
  Array
  (read-column-by-label [^Array v _]
    (vec (.getArray v)))
  (read-column-by-index [^Array v _ _]
    (vec (.getArray v))))

;; =============================================================================
;; Postgres JSON helpers
;; =============================================================================

(defn ->pgobject
  "Transforms Clojure data to a PGobject that contains the data as JSON.
   PGObject type defaults to `jsonb` but can be changed via metadata key `:pgtype`"
  [x]
  (let [pgtype (or (:pgtype (meta x)) "jsonb")]
    (doto (PGobject.)
      (.setType pgtype)
      (.setValue (uc/transit-encode x)))))

(m/=> ->pgobject
  [:=> [:cat [:or map? vector?]]
   types/pgobject?])

(defn <-pgobject
  "Transform PGobject containing `json` or `jsonb` value to Clojure data."
  [^PGobject v]
  (let [type  (.getType v)
        value (.getValue v)]
    (if (#{"jsonb" "json"} type)
      (when value
        (with-meta (uc/transit-decode value) {:pgtype type}))
      value)))

(m/=> <-pgobject
  [:=> [:cat types/pgobject?]
   :any])

(extend-protocol jdbc.prep/SettableParameter
  IPersistentMap
  (set-parameter [m ^PreparedStatement s i]
    (.setObject s i (->pgobject m)))

  IPersistentVector
  (set-parameter [v ^PreparedStatement s i]
    (.setObject s i (->pgobject v))))

(extend-protocol jdbc.rs/ReadableColumn
  PGobject
  (read-column-by-label [^PGobject v _]
    (<-pgobject v))
  (read-column-by-index [^PGobject v _2 _3]
    (<-pgobject v)))

;; =============================================================================
;; Postgres Time helpers
;; =============================================================================

(extend-protocol jdbc.rs/ReadableColumn
  Time
  (read-column-by-label [^Time v _]
    (.toLocalTime v))
  (read-column-by-index [^Time v _2 _3]
    (.toLocalTime v)))

;; =============================================================================
;; Postgres Interval helpers
;; =============================================================================

(defn ->pg-interval
  "Takes a Duration instance and converts it into a PGInterval
   instance where the interval is created as a number of seconds."
  [^Duration duration]
  (PGInterval. 0 0
    (.toDaysPart duration)
    (.toHoursPart duration)
    (.toMinutesPart duration)
    (.toSecondsPart duration)))

(extend-protocol jdbc.prep/SettableParameter
  ;; Convert durations to PGIntervals before inserting into db
  Duration
  (set-parameter [^Duration v ^PreparedStatement s ^long i]
    (.setObject s i (->pg-interval v))))

(defn <-pg-interval
  "Takes a PGInterval instance and converts it into a Duration
   instance. Ignore sub-second units."
  [^PGInterval interval]
  (-> Duration/ZERO
    (.plusSeconds (.getSeconds interval))
    (.plusMinutes (.getMinutes interval))
    (.plusHours (.getHours interval))
    (.plusDays (.getDays interval))))

(extend-protocol jdbc.rs/ReadableColumn
  ;; Convert PGIntervals back to durations
  PGInterval
  (read-column-by-label [^PGInterval v _]
    (<-pg-interval v))
  (read-column-by-index [^PGInterval v _2 _3]
    (<-pg-interval v)))
