(ns com.vadelabs.rest-core.postman
  (:require
   [camel-snake-kebab.core :as csk]
   [com.vadelabs.utils-core.interface :as uc]
   [malli.core :as m]))

(defn path-template
  [{:keys [url]}]
  (uc/str-join "/" (:path url)))

(defn request-method
  [{:keys [method]}]
  (-> method uc/str-lower uc/keywordize))

(defn path-schema
  [request]
  (let [path-params (-> request
                      path-template
                      uc/template-tags)]
    (when (seq path-params)
      (reduce
        (fn [acc param]
          (conj acc [param {} :string]))
        [:map {}]
        path-params))))

(defn ->malli
  [val]
  (case val
    "<number>" :int
    :string))

(defn query-schema
  [{:keys [url]}]
  (let [query-params (:query url)]
    (when (seq query-params)
      (reduce
        (fn [acc {:keys [key value description]}]
          (let [props (cond-> {}
                        description (assoc :doc description))]
            (conj acc [(uc/keywordize key) props (->malli value)])))
        [:map {}]
        query-params))))

(defn body-schema
  [{:keys [body]}]
  (when (seq body)
    (let [mode (-> body :mode uc/keywordize)
          val (mode body)]
      (if (string? val)
        :any
        (reduce
          (fn [acc {:keys [key value type]}]
            (let [props (cond-> {}
                          value (assoc :default value))]
              (conj acc [(uc/keywordize key) props (->malli type)])))
          [:map {}]
          val)))))

(defn ^:private request->handler
  [tag {:keys [name request item] :as req}]
  (if (seq item)
    (map (partial request->handler name) item)
    (let [path-schema (path-schema request)
          query-schema (query-schema request)
          body-schema (body-schema request)
          parameters (cond-> {}
                       path-schema (assoc :path path-schema)
                       query-schema (assoc :query query-schema)
                       body-schema (assoc :body body-schema))]
      (cond-> {:route-name (csk/->kebab-case-keyword name)
               :summary name
               :parameters parameters
               :method (request-method request)
               :path-template (path-template request)
               :response-schema {:200 :any}
               :produces #{"application/json"}
               :consumes #{"application/json"}
               :definition req
               :tags [tag]}
        path-schema (assoc :path-schema (m/schema path-schema))
        query-schema (assoc :query-schema (m/schema query-schema))
        body-schema (assoc :body-schema (m/schema body-schema))))))

(defn ^:private folders->handlers
  [{tag :name requests :item}]
  (reduce
    (fn [acc req]
      (let [handler (request->handler tag req)]
        (if (map? handler)
          (conj acc handler)
          (into acc handler))))
    []
    requests))

(defn handlers
  ([{folders :item} opts]
   (mapcat folders->handlers folders))
  ([definition]
   (handlers definition {})))
