(ns com.vadelabs.sql-core.database
  (:require
   [com.vadelabs.utils-exception.interface :as uex]
   [hikari-cp.core :as h]
   [next.jdbc.connection :as jdbc.conn])
  (:import
   [com.zaxxer.hikari HikariDataSource]))

(defn get-db-spec [{:keys [url user password dbtype host dbname port] :as params}]
  (let [jdbc-url (or url
                   (System/getenv "DATABASE_URL")
                   (try (jdbc.conn/jdbc-url
                          {:dbtype   dbtype
                           :dbname   dbname
                           :host     host
                           :port     port
                           :user     user
                           :password password})
                     (catch Exception error)))]
    (tap> {:params params})
    (if (some? jdbc-url)
      (-> params
        (dissoc :url :dbtype :dbname :host :port :user :password)
        (assoc :jdbc-url jdbc-url))
      (uex/raise
        :type uex/incorrect
        :code :incorrect-jdbc-url
        :params params
        :hint "Please provide a valid database jdbc url or db spec map.
              See https://cljdoc.org/d/com.github.seancorfield/next.jdbc/1.2.780/api/next.jdbc#get-datasource"))))

(defn connect
  [config]
  (tap> {:config config
         :db-spec (get-db-spec config)})
  (-> config get-db-spec h/make-datasource))

(defn disconnect!
  [^HikariDataSource ds]
  (when-not (.isClosed ds)
    (h/close-datasource ds)))
