(ns com.vadelabs.sql-core.adapter.postgres
  (:require
   [com.vadelabs.sql-core.adapter :refer [IAdapter]]
   [com.vadelabs.sql-core.entity :as entity]
   [com.vadelabs.sql-core.types :as types]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]
   [honey.sql :as hsql]
   [malli.core :as m]
   [next.jdbc :as jdbc]
   [next.jdbc.date-time :as jdbc.dt]
   [next.jdbc.prepare :as jdbc.prep]
   [next.jdbc.result-set :as jdbc.rs]
   [next.jdbc.types :refer [as-other]])
  (:import
   [javax.sql DataSource]
   [java.sql PreparedStatement Time Array]
   [org.postgresql.util PGobject PGInterval]
   [com.vadelabs.sql_core.entity Entity]
   [clojure.lang IPersistentMap IPersistentVector]
   [java.time Duration]))

(jdbc.dt/read-as-local)

(defn ->column-name
  "Add quoting for table name if field spec include :wrap flag"
  [entity field-name]
  (if (entity/entity-field-prop entity field-name :wrap)
    [:quote field-name]
    (-> field-name name (ustr/replace "-" "_") keyword)))

(def column-name?
  [:or :keyword
   [:tuple [:= :quote] :keyword]])

(m/=> ->column-name
  [:=> [:cat entity/entity? :keyword]
   column-name?])

;; =============================================================================
;; Postgres Array helpers
;; =============================================================================

(extend-protocol jdbc.rs/ReadableColumn
  Array
  (read-column-by-label [^Array v _]
    (vec (.getArray v)))
  (read-column-by-index [^Array v _ _]
    (vec (.getArray v))))

;; =============================================================================
;; Postgres JSON helpers
;; =============================================================================

(defn ->pgobject
  "Transforms Clojure data to a PGobject that contains the data as JSON.
   PGObject type defaults to `jsonb` but can be changed via metadata key `:pgtype`"
  [x]
  (let [pgtype (or (:pgtype (meta x)) "jsonb")]
    (doto (PGobject.)
      (.setType pgtype)
      (.setValue (uc/json-encode x)))))

(m/=> ->pgobject
  [:=> [:cat [:or map? vector?]]
   types/pgobject?])

(defn <-pgobject
  "Transform PGobject containing `json` or `jsonb` value to Clojure data."
  [^PGobject v]
  (let [type  (.getType v)
        value (.getValue v)]
    (if (#{"jsonb" "json"} type)
      (when value
        (with-meta (uc/json-decode value keyword) {:pgtype type}))
      value)))

(m/=> <-pgobject
  [:=> [:cat types/pgobject?]
   :any])

(extend-protocol jdbc.prep/SettableParameter
  IPersistentMap
  (set-parameter [m ^PreparedStatement s i]
    (.setObject s i (->pgobject m)))

  IPersistentVector
  (set-parameter [v ^PreparedStatement s i]
    (.setObject s i (->pgobject v))))

(extend-protocol jdbc.rs/ReadableColumn
  PGobject
  (read-column-by-label [^PGobject v _]
    (<-pgobject v))
  (read-column-by-index [^PGobject v _2 _3]
    (<-pgobject v)))

;; =============================================================================
;; Postgres Time helpers
;; =============================================================================

(extend-protocol jdbc.rs/ReadableColumn
  Time
  (read-column-by-label [^Time v _]
    (.toLocalTime v))
  (read-column-by-index [^Time v _2 _3]
    (.toLocalTime v)))

;; =============================================================================
;; Postgres Interval helpers
;; =============================================================================

(defn ->pg-interval
  "Takes a Duration instance and converts it into a PGInterval
   instance where the interval is created as a number of seconds."
  [^Duration duration]
  (PGInterval. 0 0
    (.toDaysPart duration)
    (.toHoursPart duration)
    (.toMinutesPart duration)
    (.toSecondsPart duration)))

(extend-protocol jdbc.prep/SettableParameter
  ;; Convert durations to PGIntervals before inserting into db
  Duration
  (set-parameter [^Duration v ^PreparedStatement s ^long i]
    (.setObject s i (->pg-interval v))))

(defn <-pg-interval
  "Takes a PGInterval instance and converts it into a Duration
   instance. Ignore sub-second units."
  [^PGInterval interval]
  (-> Duration/ZERO
    (.plusSeconds (.getSeconds interval))
    (.plusMinutes (.getMinutes interval))
    (.plusHours (.getHours interval))
    (.plusDays (.getDays interval))))

(extend-protocol jdbc.rs/ReadableColumn
  ;; Convert PGIntervals back to durations
  PGInterval
  (read-column-by-label [^PGInterval v _]
    (<-pg-interval v))
  (read-column-by-index [^PGInterval v _2 _3]
    (<-pg-interval v)))

(defn prep-join-query
  "Returns a SQL query vector to join rows from the referenced table"
  [{:keys [entity ref table-name ref-key ref-name nested]}]
  (let [ref-props        (entity/properties ref)
        ref-table        (entity/ref-entity-prop ref :table)
        ref-table-key    (keyword ref-table)
        ref-entity       (entity/field-type-prop ref :entity)
        ref-query-params (if (map? nested)
                           (get nested ref-key)
                           {})]
    (cond
      (entity/required-ref? ref-props)
      (let [ref-pk           (-> (entity/ident-field-schema ref-entity)
                               key)
            entity-ref-field (keyword (format "%s.%s" table-name ref-name))
            entity-ref-match [:= ref-pk entity-ref-field]]
        (hsql/format {:select (or (:fields ref-query-params) [:*])
                      :from   ref-table-key
                      :where  (if (some? (:where ref-query-params))
                                [:and entity-ref-match (:where ref-query-params)]
                                entity-ref-match)}
          {:quoted true}))

      (= :one-to-many (:rel-type ref-props))
      (let [target-field-name (-> (entity/ref-field-schema ref-entity entity)
                                key
                                name)
            target-column     (keyword (format "%s.%s" ref-table target-field-name))
            entity-pk         (-> (entity/ident-field-schema entity)
                                key
                                name)
            entity-pk-column  (keyword (format "%s.%s" table-name entity-pk))
            entity-ref-match  [:= target-column entity-pk-column]]
        (hsql/format (uc/assoc-some
                       {:select (or (:fields ref-query-params) [:*])
                        :from   ref-table-key
                        :where  (if (some? (:where ref-query-params))
                                  [:and entity-ref-match (:where ref-query-params)]
                                  entity-ref-match)}
                       :limit (:limit ref-query-params)
                       :offset (:offset ref-query-params)
                       :order-by (:order-by ref-query-params))
          {:quoted true}))

      (= :many-to-many (:rel-type ref-props))
      (let [query-fields       (if (some? (:fields ref-query-params))
                                 (mapv #(->> % name (format "tt.%s") keyword)
                                   (:fields ref-query-params))
                                 [:tt.*])
            join-entity        (:join ref-props)
            join-table         (-> (entity/prop join-entity :table)
                                   keyword)
            join-entity-key    (-> (entity/ref-field-schema join-entity entity)
                                   key name)
            join-ref-key       (-> (entity/ref-field-schema join-entity ref-entity)
                                   key name)
            ref-pk             (-> (entity/ident-field-schema ref-entity)
                                   key name)
            ref-pk-column      (keyword (format "tt.%s" ref-pk))
            join-ref-column    (keyword (format "jt.%s" join-ref-key))

            entity-pk          (-> (entity/ident-field-schema entity)
                                   key
                                   name)
            entity-pk-column   (keyword (format "%s.%s" table-name entity-pk))
            join-entity-column (keyword (format "jt.%s" join-entity-key))
            entity-ref-match   [:= join-entity-column entity-pk-column]]
        (hsql/format (uc/assoc-some
                       {:select query-fields
                        :from   [[join-table :jt]]
                        :join   [[ref-table-key :tt] [:= join-ref-column ref-pk-column]]
                        :where  (if (some? (:where ref-query-params))
                                  [:and entity-ref-match (:where ref-query-params)]
                                  entity-ref-match)}
                       :limit (:limit ref-query-params)
                       :offset (:offset ref-query-params)
                       :order-by (:order-by ref-query-params))
          {:quoted true})))))
(def field-params?
  [:map
   [:fields {:optional true} [:vector :keyword]]
   [:where {:optional true} vector?]
   [:limit {:optional true} :int]
   [:offset {:optional true} :int]
   [:order-by {:optional true} vector?]])

(def nested-params?
  [:or
   :boolean
   [:vector :keyword]
   [:map-of :keyword field-params?]])

(m/=> prep-join-query
  [:=> [:cat [:map
              [:entity entity/entity?]
              [:table-name :string]
              [:ref entity/schema?]
              [:ref-name :string]
              [:ref-key :keyword]
              [:nested [:maybe nested-params?]]]]
   [:vector :string]])

(defn wrap-coalesce
  "Wrap query with coalesce expression"
  [query]
  (hsql/format-expr
    [:coalesce
     [[:raw query]]
     [:inline "[]"]]))

(m/=> wrap-coalesce
  [:=> [:cat [:vector :string]]
   [:vector :string]])

(defn prep-json-query
  "Takes the join query for nested tables and wraps it with SQL query to convert nested records into JSON"
  [{:keys [ref table-name ref-name join-query]}]
  (let [temp-join-key (keyword (format "%s_%s" table-name ref-name))
        ref-props     (entity/properties ref)]
    (cond
      (entity/required-ref? ref-props)
      (hsql/format {:select [[[:row_to_json temp-join-key]]]
                    :from   [[[:nest [:raw join-query]]
                              temp-join-key]]}
        {:quoted true})

      (entity/optional-ref? ref-props)
      (-> (hsql/format {:select [[[:array_to_json [:array_agg [:row_to_json temp-join-key]]]]]
                        :from   [[[:nest [:raw join-query]]
                                  temp-join-key]]}
            {:quoted true})
        wrap-coalesce))))

(m/=> prep-json-query
  [:=> [:cat [:map
              [:ref entity/schema?]
              [:ref-name :string]
              [:table-name :string]
              [:join-query [:vector :string]]]]
   [:vector :string]])

(defn get-refs-list
  "Returns the list of ref fields. *nest* param is used to control which fields are required"
  [entity nested]
  (cond
    (true? nested)  ;; return all nested records
    (->> (entity/entity-fields entity)
      (filter (fn [[_ field]]
                (and (entity/ref? field)
                  (some? (entity/ref-entity-prop field :table))))))

    (vector? nested) ;; return only listed nested records
    (let [nested-set (set nested)]
      (->> (entity/entity-fields entity)
        (filter (fn [[field-key field]]
                  (and (contains? nested-set field-key)
                    (entity/ref? field)
                    (some? (entity/ref-entity-prop field :table)))))))

    (map? nested)   ;; fine-grained control of nested entities
    (let [nested-set (set (keys nested))]
      (->> (entity/entity-fields entity)
        (filter (fn [[field-key field]]
                  (and (contains? nested-set field-key)
                    (entity/ref? field)
                    (some? (entity/ref-entity-prop field :table)))))))

    :else []))

(m/=> get-refs-list
  [:=> [:cat entity/entity? [:maybe nested-params?]]
   [:sequential entity/entity-field-schema?]])

(defn entity-select-query
  "Build the main entity SQL query as HoneySQL DSL"
  [entity fields nested]
  (let [table-name    (entity/prop entity :table)
        table         (keyword table-name)
        default-query {:select (or fields [:*])
                       :from   [table]}
        refs          (get-refs-list entity nested)]
    (reduce (fn [query [ref-key ref]]
              (let [ref-name   (ustr/replace (name ref-key) "-" "_")
                    join-query (prep-join-query
                                 {:entity     entity
                                  :ref        ref
                                  :table-name table-name
                                  :ref-key    ref-key
                                  :ref-name   ref-name
                                  :nested     nested})
                    json-query (prep-json-query
                                 {:ref        ref
                                  :table-name table-name
                                  :ref-name   ref-name
                                  :join-query join-query})]
                (update query :select conj [[:nest [:raw json-query]] ref-key])))
      default-query
      refs)))

(m/=> entity-select-query
  [:=> [:cat entity/entity? [:maybe [:vector :keyword]] [:maybe nested-params?]]
   [:map
    [:select vector?]
    [:from vector?]]])

(defn coerce-nested-records
  "Nested records returned as JSON. This function will use nested entity spec to coerce record fields"
  [entity record]
  (let [refs (->> (entity/entity-fields entity)
               (filter (fn [[_ field]] (entity/ref? field))))]
    (reduce (fn [rec [ref-key ref]]
              (let [ref-entity (entity/field-type-prop ref :entity)]
                (update rec ref-key
                  (fn [field-val]
                    (cond
                      (sequential? field-val)
                      (mapv #(entity/cast ref-entity %) field-val)

                      (map? field-val)
                      (entity/cast ref-entity field-val)

                      :else
                      field-val)))))
      record
      refs)))

(m/=> coerce-nested-records
  [:=> [:cat entity/entity? map?]
   map?])

(defn simple-val-or-nested-entity?
  "Predicate function to check if field spec refers to a simple data type
   or json field or mandatory dependency entity"
  [field-schema]
  (or (not (entity/ref? field-schema))
    (let [props     (entity/properties field-schema)
          ref-table (entity/ref-entity-prop field-schema :table)]
      (or (not ref-table)
        (not (entity/optional-ref? props))))))

(m/=> simple-val-or-nested-entity?
  [:=> [:cat entity/schema?]
   :boolean])

(defn entity-columns
  "Collect columns names as keywords"
  [entity]
  (->> (entity/entity-fields entity)
    (filter #(simple-val-or-nested-entity? (val %)))
    (mapv key)))

(m/=> entity-columns
  [:=> [:cat entity/entity?]
   [:vector :keyword]])

(defn lift-value
  "Wraps vector or map with :lift operator to store them as json"
  [x]
  (if (or (vector? x) (map? x))
    [:lift x]
    x))

(defn prep-value
  [field-schema v]
  (let [enum?  (and (entity/ref? field-schema)
                 (some? (entity/ref-entity-prop field-schema :enum)))
        array? (= (-> field-schema entity/field-schema :type) :array)]
    (cond
      enum? (as-other v)
      array? (into-array v)
      :else (lift-value v))))

(defn prep-data
  "Builds a list of columns names and respective values based on the field schema"
  [entity data]
  (reduce-kv
    (fn [[columns values :as acc] k v]
      (if-some [field (val (entity/entity-field entity k))]
        (if (simple-val-or-nested-entity? field)
          (let [value  (prep-value field v)
                column (->column-name entity k)]
            [(conj columns column)
             (conj values value)])
          acc)
       ;; some junk in the data, noop
        acc))
    []
    data))

(defn prep-data-map [entity data]
  (reduce-kv
    (fn [acc k v]
      (if-some [field (entity/entity-field entity k)]
        (let [field-schema (val field)]
          (if (simple-val-or-nested-entity? field-schema)
            (let [value  (prep-value field-schema v)
                  column (->column-name entity k)]
              (assoc acc column value))
            acc))
       ;; some junk in the data, noop
        acc))
    {}
    data))

(defn where-clause [where-clauses eq-clauses]
  (cond
    (and (some? where-clauses) (some? eq-clauses)) [:and where-clauses eq-clauses]
    (some? eq-clauses) eq-clauses
    (some? where-clauses) where-clauses
    :else nil))

(defn pg-insert!
  "Save record in database"
  [^DataSource database entity data]
  ;; TODO deal with nested records (not JSON fields). Extract identity field if it's map
  (tap> {:ds database
         :entity entity
         :data data})
  (let [table (entity/prop entity :table)
        [columns values] (prep-data entity data)
        query (hsql/format {:insert-into table
                            :columns     columns
                            :values      [values]})]
    (try (jdbc/execute-one! database query
           {:return-keys true
            :builder-fn  jdbc.rs/as-unqualified-kebab-maps})
      (catch Exception e
        (tap> e)
        (throw e)))))

(m/=> pg-insert!
  [:=> [:cat types/connection? entity/entity? map?]
   map?])

(defn pg-insert-multi!
  "Save records in database"
  [^DataSource database entity data]
  (jdbc/with-transaction [tx database]
    (mapv (fn [record]
            (pg-insert! tx entity record))
      data)))

(m/=> pg-insert-multi!
  [:=> [:cat types/connection? entity/entity? vector?]
   vector?])

(defn pg-update!
  "Update record in database"
  [^DataSource database entity data {:fx.repo/keys [where] :as params}]
  (let [table        (entity/prop entity :table)
        eq-clauses   (some-> (prep-data-map entity params)
                       (not-empty)
                       (hsql/map=))
        where-clause (where-clause where eq-clauses)
        query        (hsql/format {:update-raw [:quote table]
                                   :set        (prep-data-map entity data)
                                   :where      where-clause})]
    (jdbc/execute-one! database query
      {:return-keys true
       :builder-fn  jdbc.rs/as-unqualified-kebab-maps})))

(m/=> pg-update!
  [:=> [:cat
        types/connection?
        entity/entity?
        map?
        [:map [:fx.repo/where {:optional true} vector?]]]
   map?])

(defn pg-delete!
  "Delete record from database"
  [^DataSource database entity {:fx.repo/keys [where] :as params}]
  (let [table        (entity/prop entity :table)
        eq-clauses   (some-> (prep-data-map entity params)
                       (not-empty)
                       (hsql/map=))
        where-clause (where-clause where eq-clauses)
        query        (hsql/format {:delete-from (keyword table)
                                   :where       where-clause})]
    (jdbc/execute-one! database query
      {:return-keys true
       :builder-fn  jdbc.rs/as-unqualified-kebab-maps})))

(m/=> pg-delete!
  [:=> [:cat
        types/connection?
        entity/entity?
        [:map [:fx.repo/where {:optional true} vector?]]]
   map?])

(defn pg-query-single
  "Get single record from the database"
  [^DataSource database entity {:fx.repo/keys [fields where nested] :as params}] ;; TODO add exclude parameter to filter fields
  (let [eq-clauses (some-> (prep-data-map entity params)
                     (not-empty)
                     (hsql/map=))
        where-map  {:where (where-clause where eq-clauses)}
        select-map (entity-select-query entity fields nested)
        query      (-> select-map
                     (merge where-map)
                     (hsql/format {:quoted true})
                     (uc/tap-> "Find query"))
        record     (jdbc/execute-one! database query
                     {:return-keys true
                      :builder-fn  jdbc.rs/as-unqualified-kebab-maps})]
    (println "nested" nested)
    (if (some? nested)
      (coerce-nested-records entity record)
      record)))

(m/=> pg-query-single
  [:=> [:cat
        types/connection?
        entity/entity?
        [:map
         [:fx.repo/fields {:optional true} [:vector :keyword]]
         [:fx.repo/where {:optional true} vector?]
         [:fx.repo/nested {:optional true} nested-params?]]]
   [:maybe map?]])

(defn pg-query
  "Return multiple records from the database"
  [^DataSource database entity {:fx.repo/keys [fields where order-by limit offset nested] :as params}]
  (let [eq-clauses (some-> (prep-data-map entity params)
                     (not-empty)
                     (hsql/map=))
        rest-map   (uc/assoc-some
                     {}
                     :where (where-clause where eq-clauses)
                     :limit limit
                     :offset offset
                     :order-by order-by)
        select-map (entity-select-query entity fields nested)
        query      (-> select-map
                     (merge rest-map)
                     (hsql/format {:quoted true}))
        records    (jdbc/execute! database query
                     {:return-keys true
                      :builder-fn  jdbc.rs/as-unqualified-kebab-maps})]
    (if (some? nested)
      (mapv #(coerce-nested-records entity %) records)
      records)))

(m/=> pg-query
  [:=> [:cat
        types/connection?
        entity/entity?
        [:map
         [:fx.repo/fields {:optional true} [:vector :keyword]]
         [:fx.repo/where {:optional true} vector?]
         [:fx.repo/nested {:optional true} nested-params?]
         [:fx.repo/order-by {:optional true} vector?]
         [:fx.repo/limit {:optional true} :int]
         [:fx.repo/offset {:optional true} :int]]]
   [:maybe [:vector map?]]])

(defn pg-save!
  [& args])

(defn init
  [database]
  (extend-protocol IAdapter
    Entity
    (save! [entity data]
      (pg-save! database entity data))

    (insert! [entity data]
      (pg-insert! database entity data))

    (insert-multi! [entity data]
      (pg-insert-multi! database entity data))

    (update! [entity data params]
      (pg-update! database entity data params))

    (delete! [entity params]
      (pg-delete! database entity params))

    (query-single [entity params]
      (pg-query-single database entity params))

    (query
      ([entity]
       (pg-query database entity {}))
      ([entity params]
       (pg-query database entity params)))))
