(ns com.vadelabs.adapter-postgres.interface
  (:require
   [com.vadelabs.adapter-core.interface :as ac]
   [com.vadelabs.adapter-postgres.pathom]
   [com.vadelabs.sql-core.interface :as sc]
   [com.vadelabs.utils-core.interface :as uc]
   [com.vadelabs.utils-str.interface :as ustr]))

(def types-map
  {:attribute.type/bigdec :double
   :attribute.type/bigint :bigint
   :attribute.type/boolean :boolean
   :attribute.type/double :double
   :attribute.type/enum :enum
   :attribute.type/float :double
   :attribute.type/secret :string
   :attribute.type/datetime :time-tz
   :attribute.type/keyword :string
   :attribute.type/long :bigint
   :attribute.type/map :jsonb
   :attribute.type/ref :ref
   :attribute.type/string :string
   :attribute.type/symbol :string
   :attribute.type/tuple :array
   :attribute.type/uuid :uuid
   :attribute.type/uri :string})

(defn ->raw-spec
  [remote-ns remote-entity attributes]
  (into
    [:spec {:table (uc/prefix-keyword "." remote-ns remote-entity)}]
    (->> attributes
      (map (fn [{:attribute/keys [remote-key identity primary-key remote-type target cardinality]}]
             (let [ref? (= remote-type :ref)
                   many? (= cardinality :many)
                   one? (not many?)
                   enum? (= remote-type :enum)
                   remote-type (cond
                                 ref? (uc/keywordize remote-ns (namespace target))
                                 enum? (uc/keywordize remote-ns remote-entity remote-key)
                                 :else remote-type)
                   props (cond-> {}
                           identity (assoc :identity true)
                           primary-key (assoc :identity true)
                           (and ref? many?) (assoc :rel-type :one-to-many)
                           (and ref? one?) (assoc :rel-type :one-to-one))]
               [remote-key props remote-type]))))))

(defn attributes->entities
  [attributes]
  (let [enum-spec (->> attributes
                    (filter (fn [{:attribute/keys [remote-type]}]
                              (= remote-type :enum)))
                    (reduce (fn [acc {:attribute/keys [remote-ns remote-entity remote-key options]}]
                              (assoc acc (uc/keywordize remote-ns remote-entity remote-key)
                                [:spec {:enum (apply ustr/format "%s.%s_%s" (map uc/namify [remote-ns remote-entity remote-key]))
                                        :values (mapv uc/namify options)}]))
                      {}))]
    (->> attributes
      (group-by (juxt :attribute/remote-ns :attribute/remote-entity))
      (reduce-kv (fn [acc [remote-ns remote-entity] attributes]
                   (assoc acc (uc/keywordize remote-ns remote-entity)
                     (->raw-spec remote-ns remote-entity attributes)))
        enum-spec)
      sc/register-entities)))

(def ^:private adapter-type :adapter.type/postgres)

(defmethod ac/initiate-aenv! adapter-type
  [genv {:adapter/keys [type id]}]
  (let [aenv (get-in genv [type id])
        db-conn (sc/connect aenv)
        aenv (assoc aenv :db-conn db-conn)]
    (sc/create-db-schema! aenv)
    (sc/create-migrations-table! aenv)
    (-> genv
      (assoc-in [type id :db-conn] db-conn))))

(defmethod ac/halt-aenv! adapter-type
  [genv {:adapter/keys [type id]}]
  (let [aenv (get-in genv [type id])]
    (sc/disconnect! aenv)))

(defmethod ac/attributes adapter-type
  [genv {:adapter/keys [type nspace id attributes] :or {attributes []} :as adapter}]
  (let [attributes (mapv (fn [{:attribute/keys [remote-ns remote-entity remote-key
                                                remote-type local-key local-type] :as attribute}]
                           (ac/prepare-attribute adapter
                             (assoc attribute
                               :attribute/remote-ns (or remote-ns nspace)
                               :attribute/remote-entity (or remote-entity (-> local-key namespace keyword))
                               :attribute/remote-key (or remote-key (-> local-key name keyword))
                               :attribute/remote-type (or remote-type (local-type types-map)))))
                     attributes)
        attributes-map (ac/attributes-map attributes)
        entities (attributes->entities attributes)]
    (tap> {:entities entities})
    (-> genv
      (assoc-in [type id :entities] entities)
      (assoc-in [type id :attributes] attributes)
      (assoc-in [type id :attributes-map] attributes-map))))

(defmethod ac/sync-schema! adapter-type
  [genv {:adapter/keys [type id]}]
  (let [{:keys [db-conn entities]} (get-in genv [type id])]
    (sc/apply-migrations! {:database db-conn :entities (into #{} (vals entities))})
    genv))

(def adapter
  {:adapter/id (uc/uuid ::adapter)
   :adapter/nspace :pg
   :adapter/identifier :postgres
   :adapter/type adapter-type
   :adapter/qualify-attributes true
   :adapter/config  {:url "jdbc:postgresql://localhost:6432/vadedb?user=vadeuser&password=vadepassword"}})

(def datasources
  [{:datasource/id (uc/uuid ::datasource)
    :datasource/display-name "PostgreSQL"
    :datasource/description "The World's Most Advanced Open Source Relational Database"
    :datasource/slug "postgresql"
    :datasource/icon "https://www.postgresql.org/media/img/about/press/elephant.png"
    :datasource/preview "https://1000logos.net/wp-content/uploads/2020/08/PostgreSQL-Logo-1024x640.png"
    :datasource/adapter adapter
    :datasource/attributes []
    :datasource/actions []
    :datasource/categories []
    :datasource/collections []}])

(comment
  (require '[com.vadelabs.datasource-studio.interface :as dstudio])
  (attributes->entities dstudio/attributes)

  (ac/initiate! {} [(assoc adapter :adapter/attributes [{:attribute/local-key :customer/id,
                                                         :attribute/local-type :attribute.type/string,
                                                         :attribute/group :customer}
                                                        {:attribute/local-key :customer/company-name,
                                                         :attribute/local-type :attribute.type/string,
                                                         :attribute/group :customer}
                                                        {:attribute/local-key :customer/user-id,
                                                         :attribute/local-type :attribute.type/string,
                                                         :attribute/group :customer}
                                                        {:attribute/local-key :customer/user-name,
                                                         :attribute/local-type :attribute.type/string,
                                                         :attribute/group :customer}
                                                        {:attribute/local-key :customer/last-login-date,
                                                         :attribute/local-type :attribute.type/string,
                                                         :attribute/group :customer}
                                                        {:attribute/local-key :customer/invoices,
                                                         :attribute/local-type :attribute.type/ref,
                                                         :attribute/target :invoice/id,
                                                         :attribute/cardinality :many,
                                                         :attribute/group :customer}
                                                        {:attribute/local-key :invoice/id,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/customer,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/reference,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/publish-as,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/terms,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/amounts-are,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/currency,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/online-payments,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/branding,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :invoice/batch,
                                                         :attribute/local-type :attribute.type/long}
                                                        {:attribute/local-key :invoice/notes,
                                                         :attribute/local-type :attribute.type/string,
                                                         :attribute/cardinality :many}
                                                        {:attribute/local-key :invoice/items,
                                                         :attribute/local-type :attribute.type/ref,
                                                         :attribute/target :item/id,
                                                         :attribute/cardinality :many}
                                                        {:attribute/local-key :item/id,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/description,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/quantity,
                                                         :attribute/local-type :attribute.type/double}
                                                        {:attribute/local-key :item/unit-price,
                                                         :attribute/local-type :attribute.type/double}
                                                        {:attribute/local-key :item/discount,
                                                         :attribute/local-type :attribute.type/double}
                                                        {:attribute/local-key :item/account,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/tax-rate,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/tax-amount,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/track-one,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/track-two,
                                                         :attribute/local-type :attribute.type/string}
                                                        {:attribute/local-key :item/amount,
                                                         :attribute/local-type :attribute.type/double}])])
  :rcf)
