(ns com.vadelabs.rest-core.main
  (:require
   [com.vadelabs.rest-core.context :as tc]
   [com.vadelabs.rest-core.interceptors :as interceptors]
   [com.vadelabs.utils-core.interface :as uc]))

(defn ^:private resolve-instance
  [http-instance]
  (cond
    (map? http-instance) http-instance
    (var? http-instance) (deref http-instance)
    (fn? http-instance) (http-instance)
    :else http-instance))

(defn ^:private enrich-handlers
  [handlers {:keys [produces consumes]}]
  (map
    (fn [handler]
      (-> handler
        (update :produces #(or % produces))
        (update :consumes #(or % consumes))))
    handlers))

(defrecord ^:private HttpInstance
  [base-url handlers interceptors opts])

(defn ^:private make-http-instance
  [base-url handlers {:keys [interceptors] :as opts}]
  (->HttpInstance base-url handlers interceptors (dissoc opts :interceptors)))

(defn ^:private find-handler
  [handlers route-name]
  (or (first (filter #(= (keyword route-name) (:route-name %)) handlers))
    (throw (ex-info (str "Could not find route " route-name)
             {:handlers   handlers
              :route-name route-name}))))

(defn bootstrap
  [base-url handlers & [opts]]
  (make-http-instance base-url (enrich-handlers handlers opts) opts))

(defn handler-for
  [http-instance route-name]
  (let [{:keys [handlers]} (resolve-instance http-instance)]
    (find-handler handlers route-name)))

(defn url-for
  ([http-instance route-name]
   (url-for http-instance route-name {}))
  ([http-instance route-name params]
   (let [{:keys [base-url handlers]} (resolve-instance http-instance)
         {:keys [path-template]} (find-handler handlers route-name)
         params params]
     (->> params
       (uc/template-render path-template)
       (uc/urlify base-url)))))

(defn request-for
  ([http-instance route-name]
   (request-for http-instance route-name {}))
  ([http-instance route-name params]
   (let [{:keys [handlers interceptors] :as http-instance} (resolve-instance http-instance)
         handler (find-handler handlers route-name)
         ctx (tc/enqueue* {} (conj interceptors
                               interceptors/request-only-handler))]
     (:request (tc/execute
                 (assoc ctx
                   :url-for (partial url-for http-instance)
                   :request (or (:rc/request params) {})
                   :handler handler
                   :params params))))))

(defn response-for
  ([http-instance route-name]
   (response-for http-instance route-name {}))
  ([http-instance route-name params]
   (let [{:keys [handlers interceptors] :as http-instance} (resolve-instance http-instance)
         handler (find-handler handlers route-name)
         ctx (tc/enqueue* {} interceptors)]
     (:response (tc/execute
                  (assoc ctx
                    :url-for (partial url-for http-instance)
                    :request (or (:rc/request params) {})
                    :params params
                    :handler handler))))))

(defn explore
  ([http-instance]
   (let [{:keys [handlers]} (resolve-instance http-instance)]
     (mapv (juxt :route-name :summary) handlers)))
  ([http-instance route-name]
   (let [{:keys [handlers]} (resolve-instance http-instance)
         {:keys [route-name summary method deprecated?]} (find-handler handlers route-name)]
     (cond-> {:summary summary
              :route-name route-name
              :method method
              :parameters {}
              :returns {}}
       deprecated? (assoc :deprecated? true)))))

(defn handlers
  [http-client]
  (-> http-client
    resolve-instance
    :handlers))

(defn update-handler
  "Update a handler in the martian record with the provided route-name
   e.g. add route-specific interceptors:
   (update-handler m :load-pet assoc :interceptors [an-interceptor])"
  [m route-name update-fn & update-args]
  (update (resolve-instance m)
    :handlers #(mapv (fn [handler]
                       (if (= (keyword route-name) (:route-name handler))
                         (apply update-fn handler update-args)
                         handler))
                 %)))
