(ns com.vadelabs.adapter-core.interface
  (:require
   [camel-snake-kebab.core :as csk]
   [com.vadelabs.utils-core.interface :as uc]
   [com.wsscode.pathom.viz.ws-connector.core :as pvc]
   [com.wsscode.pathom.viz.ws-connector.pathom3 :as p.connector]
   [com.wsscode.pathom3.connect.built-in.plugins :as pbip]
   [com.wsscode.pathom3.connect.indexes :as pci]
   [com.wsscode.pathom3.connect.operation :as pco]
   [com.wsscode.pathom3.plugin :as p.plugin]
   [malli.core :as m]
   [com.vadelabs.adapter-core.interface :as ac]))

(defn ^:private dispatch-func
  ([_genv {:adapter/keys [type]}]
   type))

(defmulti prepare-aenv
  #'dispatch-func)

(defmulti initiate-aenv!
  #'dispatch-func)

(defmulti halt-aenv!
  #'dispatch-func)

(defmulti default-actions
  (fn [{:adapter/keys [type]} _attributes]
    type))

(defmulti actions
  #'dispatch-func)

(defmulti attributes
  #'dispatch-func)

(defmulti sync-schema!
  #'dispatch-func)

(defmulti pathom-op
  (fn [{:keys [dialect]} {:action/keys [type]}]
    [dialect type]))

(defn ^:private register-pathom
  [genv {:adapter/keys [type id]}]
  (let [{:keys [actions] :as aenv} (get-in genv [type id])]
    (->> actions
      (mapv (partial pathom-op aenv))
      (pci/register genv))))

(defn initiate!
  "Initiates all adapters and combines them into a global pathom environment"
  [{:keys [parser] :as gopts} adapters]
  (let [adapters (if (map? adapters) [adapters] adapters)
        genv  (reduce (fn [aenv adapter]
                        (-> aenv
                          (prepare-aenv adapter)
                          (attributes adapter)
                          (actions adapter)
                          (initiate-aenv! adapter)
                          (sync-schema! adapter)
                          (register-pathom adapter)))
                gopts
                adapters)
        genv (cond-> genv
               (p.plugin/register pbip/mutation-resolve-params)
               (assoc :com.wsscode.pathom3.error/lenient-mode? true ::adapters adapters)
               parser (p.connector/connect-env {::pvc/parser-id parser}))]
    genv))

(defn halt!
  "Halts the datasource adapter if possible"
  [{::keys [adapters] :as genv}]
  (for [adapter adapters]
    (halt-aenv! genv adapter)))

(defn attributes-map
  [attrs]
  (->> attrs
    (mapcat (fn [{:attribute/keys [local-key qualified-key] :as attr}]
              [[local-key attr]
               (when qualified-key [qualified-key attr])]))
    (into {})))

(defmethod sync-schema! :default
  [genv _]
  genv)

(defmethod halt-aenv! :default
  [genv _]
  genv)

(defmethod prepare-aenv :default
  [genv {:adapter/keys [qualify-attributes nspace provider type id config]}]
  (cond-> genv
    nspace (assoc-in [type id :migrations-table] (uc/prefix-keyword "." nspace :migrations-table))
    nspace (assoc-in [type id :nspace] nspace)
    type (assoc-in [type id :dialect] (-> type name keyword))
    provider (assoc-in [type id :provider] provider)
    (seq config) (assoc-in [type id :config] (or config {}))
    (boolean? qualify-attributes) (assoc-in [type id :qualify-attributes] qualify-attributes)))

(defn prepare-attribute
  [{:adapter/keys [qualify-attributes nspace]}
   {:attribute/keys [id local-key local-type cardinality options default description target group
                     unique primary-key remote-ns remote-entity remote-key remote-type]}]
  (let [id (or id (uc/uuid local-key))
        qualified-key (if qualify-attributes (uc/keywordize nspace local-key) local-key)
        qualified-target (when target (if qualify-attributes (uc/keywordize nspace target) target))
        cardinality (or cardinality :one)
        group (or group :other)]
    (cond->
      {:attribute/id id
       :attribute/local-key local-key
       :attribute/local-type local-type
       :attribute/group group
       :attribute/qualified-key qualified-key
       :attribute/cardinality cardinality
       :attribute/description (or description "")}
      (boolean? unique) (assoc :attribute/unique unique)
      remote-ns (assoc :attribute/remote-ns remote-ns)
      remote-entity (assoc :attribute/remote-entity remote-entity)
      remote-key (assoc :attribute/remote-key remote-key)
      remote-type (assoc :attribute/remote-type remote-type)
      (seq options) (assoc :attribute/options options)
      default (assoc :attribute/default default)
      primary-key (assoc :attribute/primary-key primary-key)
      target (assoc :attribute/target target)
      qualified-target (assoc :attribute/qualified-target qualified-target))))

(defmethod attributes :default
  [genv {:adapter/keys [type id attributes] :or {attributes []} :as adapter}]
  (let [attributes (mapv (partial prepare-attribute adapter) attributes)
        attributes-map (attributes-map attributes)]
    (-> genv
      (assoc-in [type id :attributes] attributes)
      (assoc-in [type id :attributes-map] attributes-map))))

(defmethod default-actions :default
  [{:adapter/keys [nspace type]} attributes]
  (let [prefix-fn (partial uc/keywordize nspace)
        entity (-> type name keyword)]
    [{:action/identifier (prefix-fn :insert!)
      :action/entity entity
      :action/type :insert
      :action/params []
      :action/output [:tempids]}]))

(defmethod actions :default
  [genv {:adapter/keys [type id] :as adapter}]
  (let [{:keys [attributes]} (get-in genv [type id])
        actions (ac/default-actions adapter attributes)]
    (-> genv
      (assoc-in [type id :actions] actions))))

(defn attributes->pathom-query
  ([attrs]
   (reduce
     (fn [acc {:attribute/keys [qualified-key qualified-target local-key target]}]
       (let [qualified-key (or qualified-key local-key)
             qualified-target (or qualified-target target)]
         (cond-> acc
           qualified-target (conj {qualified-key [qualified-target]})
           (not qualified-target) (conj qualified-key))))
     []
     attrs)))

(defmulti ^:private ->attributes
  :type)

(defmethod ->attributes :default
  [{:keys [type]}]
  type)

(defmethod ->attributes :map
  [{:keys [properties keys]}]
  (let [{:keys [name]} properties]
    (reduce-kv
      (fn [acc remote-key v]
        (let [{:keys [optional description]} (:properties v)
              remote-type (->attributes (:value v))
              local-key (uc/nspacify name (csk/->kebab-case-keyword remote-key))
              attribute (cond-> {:attribute/id (uc/uuid local-key)
                                 :attribute/local-key local-key
                                 :attribute/local-type (uc/nspacify :attribute.type remote-type)
                                 :attribute/remote-key remote-key
                                 :attribute/remote-type remote-type
                                 :attribute/group name}
                          optional (assoc :attribute/optional optional)
                          description (assoc :attribute/description description))]
          (conj acc attribute)))
      []
      keys)))

(defn malli-schema->attributes
  [schema]
  (-> schema m/ast ->attributes))

(defn ->malli-schema
  [remote-ns remote-entity attributes]
  (into [:spec {:table (uc/keywordize remote-ns remote-entity)}]
    (->> attributes
      (mapv (fn [{:attribute/keys [remote-key remote-type]}]
              [(uc/keywordize remote-ns remote-entity remote-key) {} remote-type])))))

(defn attributes->malli-schema
  [attributes]
  (->> attributes
    (group-by (juxt :attribute/remote-ns :attribute/remote-entity))
    (reduce-kv (fn [acc [remote-ns remote-entity] attributes]
                 (assoc acc (uc/keywordize remote-ns remote-entity)
                   (->malli-schema remote-ns remote-entity attributes)))
      {})))

(defn mutation
  [{:action/keys [identifier params output]} mutation-fn]
  (pco/mutation
    {::pco/op-name (uc/symbolize identifier)
     ::pco/mutate mutation-fn
     ::pco/params params
     ::pco/output output}))

(defn query
  [{:action/keys [identifier params output]} resolver-fn]
  (pco/resolver
    (cond-> {::pco/op-name (uc/symbolize identifier)
             ::pco/resolve resolver-fn
             ::pco/output [{identifier output}]}
      (seq params) (assoc ::pco/params params))))

(defn query-single
  [{:action/keys [identifier input params output]} resolver-fn]
  (pco/resolver
    (cond-> {::pco/op-name (uc/symbolize identifier)
             ::pco/resolve resolver-fn}
      (seq input) (assoc ::pco/input input)
      (seq params) (assoc ::pco/params params)
      (seq output) (assoc ::pco/output output))))
