(ns com.vadelabs.rest-core.openapi
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.walk :as cwalk]
   [com.vadelabs.utils-core.interface :as uc]
   [malli.core :as m]))

(defmulti ->malli :type)

(defmethod ->malli "string"
  [{:keys [format]}]
  (case format
    "uuid" :uuid
    :string))

(defmethod ->malli "boolean"
  [_]
  :boolean)

(defmethod ->malli "number"
  [_]
  :int)

(defmethod ->malli "integer"
  [_]
  :int)

(defmethod ->malli "object"
  [{:keys [required additionalProperties properties]}]
  (let [required? (->> required (map uc/keywordize) (into #{}))
        closed? (false? additionalProperties)
        props {:closed closed?}
        fields (reduce-kv
                 (fn [acc field-key field-schema]
                   (let [optional? (not (required? field-key))]
                     (conj acc [field-key
                                {:optional optional?}
                                (->malli field-schema)])))
                 []
                 properties)]
    (into [:map props] fields)))

(defmethod ->malli "array"
  [{:keys [items] :as schema}]
  (cond
    (vector? items) (into [:tuple] (map ->malli items))
    (map? items) [:vector (->malli items)]
    :else (throw (ex-info "Not supported" {:schema schema
                                           :reason :array-items}))))

(defmethod ->malli :default
  [{:keys [properties] :as schema}]
  (if (seq properties)
    (->malli (assoc schema :type "object"))
    :any))

(defn ^:private openapi-ref
  [param]
  (:$ref param))

(defn ^:private lookup-ref
  [components reference]
  (->> (uc/str-split reference #"/")
    (drop 2)
    (mapv uc/keywordize)
    (get-in components)))

(defn ^:private resolve-reference
  [{:keys [components]} item]
  (let [ref (openapi-ref item)]
    (if ref
      (lookup-ref components ref)
      item)))

(defn ^:private resolve-references
  [openapi-spec obj]
  (cwalk/postwalk (partial resolve-reference openapi-spec) obj))

(def ^:private http-methods
  #{:get :put :post :delete})

(defn ^:private parameter->schema
  [params]
  (let [schema (reduce
                 (fn [acc {:keys [name required description example schema]}]
                   (let [props (cond-> {}
                                 (not required) (assoc :optional true)
                                 description (assoc :doc description)
                                 example (assoc :example example))]
                     (conj acc [(uc/keywordize name) props (->malli schema)])))
                 [:map {}]
                 params)]
    (m/schema schema)))

(defn ^:private parameters->schema
  [parameters]
  (->> parameters
    (uc/group-by :in)
    (reduce-kv
      (fn [acc in params]
        (assoc acc (uc/keywordize in) (parameter->schema params)))
      {})))

(defn ^:private responses->schema
  [responses]
  (reduce-kv
    (fn [acc status-code response]
      (let [produces (-> response :content keys first)
            path [:content produces :schema]
            response-schema (get-in response path)
            malli-schema (m/schema (->malli response-schema))]
        (assoc acc status-code malli-schema)))
    {}
    responses))

(defn ^:private response-content-types
  [responses]
  (->> responses
    vals
    (map (fn [resp] (-> resp :content keys first uc/stringify)))
    (into #{})))

(defn ^:private request-content-types
  [request-body]
  (-> request-body
    :content
    keys
    first))

(defn ^:private request-body->schema
  [request-body]
  (let [consumes (-> request-body :content keys first)
        path [:content consumes :schema]
        request-schema (get-in request-body path)
        malli-schema (m/schema (->malli request-schema))]
    malli-schema))

(defn ^:private method->handler
  [path-kw method definition]
  (let [{:keys [operationId parameters responses summary tags requestBody]} definition
        {:keys [path query]} (parameters->schema parameters)
        body (request-body->schema requestBody)
        parameters (cond-> {}
                     path (assoc :path path)
                     query (assoc :query query)
                     body (assoc :body body))]
    (cond-> {:route-name (csk/->kebab-case-keyword operationId)
             :method method
             :parameters parameters
             :path-template (uc/namify path-kw)
             :produces (response-content-types responses)
             :response-schema (responses->schema responses)
             :summary summary
             :tags tags
             :definition definition}
      path (assoc :path-schema path)
      query (assoc :query-schema query)
      requestBody (assoc :body-schema (request-body->schema requestBody)
                    :consumes #{(request-content-types requestBody)}))))

(defn ^:private methods->handlers
  [path-kw methods]
  (reduce-kv
    (fn [acc method definition]
      (if (http-methods method)
        (conj acc (method->handler path-kw method definition))
        acc))
    []
    methods))

(defn ^:private paths->handlers
  [paths]
  (reduce-kv
    (fn [acc path-kw methods]
      (into acc (methods->handlers path-kw methods)))
    []
    paths))

(defn handlers
  [{:keys [paths components] :as openapi-spec}]
  (let [{:keys [schemas parameters responses]} components
        responses (resolve-references openapi-spec responses)
        parameters (resolve-references openapi-spec parameters)
        schemas (resolve-references openapi-spec schemas)
        paths (resolve-references {:components {:parameters parameters
                                                :responses responses
                                                :schemas schemas}
                                   :paths paths}
                paths)]
    (paths->handlers paths)))

(defn base-url
  [{:keys [servers]}]
  (-> servers first :url))
