(ns com.vadelabs.datasource-core.sse
  (:require
   [clojure.core.async :as a]
   [com.vadelabs.utils-core.interface :as uc]
   [hato.client :as client]
   [clj-http.client :as hc])
  (:import (java.io InputStream)))

#_(hc/request
    {:headers {"Authorization"
               "Bearer sk_test_a2V5XzAxR1RFQkYwRjVFMlpHNERZSERIUkVSVDZFLGM2azF1UHFUb1VDdFBKUHNTSjlFdVZpVzI"},
     :method :post
     :url "https://api.workos.com/passwordless/sessions",
     :form-params {:email "pragyan@vadelabs.com", :type "MagicLink"}
     :as :auto})

(def event-mask
  (re-pattern (str "(?s).+?\n\n")))

(defn deliver-events
  [events {:keys [on-next]}]
  (when on-next
    (a/go
      (loop []
        (let [event (a/<! events)]
          (when (not= :done event)
            (on-next event)
            (recur)))))))

(defn ^:private parse-event [raw-event]
  (let [data-idx (uc/str-index-of raw-event "{")
        done-idx (uc/str-index-of raw-event "[DONE]")]
    (if done-idx
      :done
      (-> (subs raw-event data-idx)
        (uc/json-decode)))))

(defn calc-buffer-size
  "Buffer size should be at least equal to max_tokens
  or 16 (the default in openai as of 2023-02-19)
  plus the [DONE] terminator"
  [{:keys [max_tokens]
    :or {max_tokens 16}}]
  (inc max_tokens))

(defn sse-events
  "Returns a core.async channel with events as clojure data structures.
  Inspiration from https://gist.github.com/oliyh/2b9b9107e7e7e12d4a60e79a19d056ee"
  [{:keys [request params]}]
  (let [event-stream ^InputStream (:body (client/request (merge request
                                                           params
                                                           {:as :stream})))
        buffer-size (calc-buffer-size params)
        events (a/chan (a/sliding-buffer buffer-size) (map parse-event))]
    (a/thread
      (loop [data nil]
        (let [byte-array (byte-array (max 1 (.available event-stream)))
              bytes-read (.read event-stream byte-array)]

          (if (neg? bytes-read)

            ;; Input stream closed, exiting read-loop
            (.close event-stream)

            (let [data (str data (slurp byte-array))]
              (if-let [es (not-empty (re-seq event-mask data))]
                (if (every? true? (map #(a/>!! events %) es))
                  (recur (uc/str-replace data event-mask ""))

                  ;; Output stream closed, exiting read-loop
                  (.close event-stream))

                (recur data)))))))
    events))

(defn sse-request
  "Process streamed results.
  If on-next callback provided, then read from channel and call the callback.
  Returns a response with the core.async channel as the body"
  [{:keys [params] :as ctx}]
  (let [events (sse-events ctx)]
    (deliver-events events params)
    {:status 200
     :body events}))
