(ns com.vadelabs.datasource-core.interceptors
  (:require
   [clojure.walk :as cwalk]
   [com.vadelabs.rest-core.interface :as rc]
   [com.vadelabs.utils-core.interface :as uc]
   #?@(:clj [[clojure.edn :as edn]
             [com.vadelabs.datasource-core.sse :as sse]
             [hato.client :as client]]
       :cljs [[cljs.reader :as edn]])
   [com.vadelabs.utils-core.string :as ustr]
   [malli.transform :as mt]
   [malli.core :as m]))

(defn ^:private insert-or-merge
  [m k v]
  (cond
    (get m k) (update m k #(merge v %))
    (not-empty v) (assoc m k v)
    :else m))

(defn ^:private create-only
  [m k v]
  (if (get m k)
    m
    (assoc m k v)))

(defn coerce-data
  [handler schema-key params opts]
  (when-let [schema (get handler schema-key)]
    (m/decode schema params (mt/transformer
                              mt/default-value-transformer
                              mt/strip-extra-keys-transformer
                              mt/json-transformer))))

;;;;;;;;;;;;;;;;;;
;; INTERCEPTORS ;;
;;;;;;;;;;;;;;;;;;

(def request-only-handler
  {:name :dci/request-only-handler
   :leave rc/remove-stack})

(def keywordize-params
  {:name :dci/keywordize-params
   :enter (fn [ctx] (update ctx :params cwalk/keywordize-keys))})

(def set-method
  {:name :dci/method
   :enter (fn [{:keys [handler] :as ctx}]
            (update ctx :request create-only :method (:method handler)))})

(def set-query-params
  {:name :dci/set-query-params
   :enter (fn [{:keys [handler params opts] :as ctx}]
            (update ctx :request insert-or-merge :query-params (coerce-data handler :query-schema params opts)))})

(def set-body-params
  {:name :dci/set-body-params
   :enter (fn [{:keys [handler params opts] :as ctx}]
            (update ctx :request insert-or-merge :body (coerce-data handler :body-schema params opts)))})

(def set-form-params
  {:name :dci/set-form-params
   :enter (fn [{:keys [handler params opts] :as ctx}]
            (update ctx :request insert-or-merge :form-params (coerce-data handler :form-schema params opts)))})

(def set-header-params
  {:name :dci/set-header-params
   :enter (fn [{:keys [handler params opts] :as ctx}]
            (update ctx :request insert-or-merge :headers (coerce-data handler :headers-schema params opts)))})

(def set-url
  {:name :dci/set-url
   :enter (fn [{:keys [params url-for handler opts] :as ctx}]
            (update ctx :request create-only :url (url-for (:route-name handler) (coerce-data handler :path-schema params opts))))})

(def enqueue-route-interceptors
  {:name :dci/enqueue-route-interceptors
   :enter (fn [{:keys [handler] :as ctx}]
            (let [{:keys [interceptors]} handler]
              (rc/enqueue-route-interceptors ctx interceptors)))})

(def default-encoders
  {"application/transit+msgpack" {:encode #(uc/transit-encode % {:type :msgpack})
                                  :decode #(uc/transit-decode % {:type :msgpack})
                                  :as :byte-array}
   "application/transit+json" {:encode #(uc/transit-encode % {:type :json})
                               :decode #(uc/transit-decode % {:type :json})}
   "application/edn" {:encode pr-str
                      :decode edn/read-string}
   "application/json" {:encode uc/json-encode
                       :decode uc/json-decode}})

(defn choose-content-type [encoders options]
  (some (set options) (keys encoders)))

(def auto-encoder
  {:encode identity
   :decode identity
   :as :auto})

(defn find-encoder [encoders content-type]
  (if (ustr/blank? content-type)
    auto-encoder
    (loop [encoders encoders]
      (let [[ct encoder] (first encoders)]
        (cond
          (not content-type) auto-encoder

          (not encoder) auto-encoder

          (ustr/includes? content-type ct) encoder

          :else
          (recur (rest encoders)))))))

(def encode-request-body
  {:name :dci/encode-request-body
   :encodes (keys default-encoders)
   :enter (fn [{:keys [request handler] :as ctx}]
            (let [content-type (and (:body request)
                                 (not (get-in request [:headers "Content-Type"]))
                                 (choose-content-type default-encoders (:consumes handler)))
                  {:keys [encode]} (find-encoder default-encoders content-type)]
              (cond-> ctx
                (get-in ctx [:request :body]) (update-in [:request :body] encode)
                content-type (assoc-in [:request :headers "Content-Type"] content-type))))})

(def encode-response-body
  {:name :dci/encode-response-body
   :decodes (keys default-encoders)
   :enter (fn [{:keys [request handler] :as ctx}]
            (let [content-type (and (not (get-in request [:headers "Accept"]))
                                 (choose-content-type default-encoders (:produces handler)))
                  {:keys [as] :or {as :text}} (find-encoder default-encoders content-type)]
              (cond-> (assoc-in ctx [:request :as] as)
                content-type (assoc-in [:request :headers "Accept"] content-type))))

   :leave (fn [{:keys [response] :as ctx}]
            (assoc ctx :response
              (let [content-type (and (:body response)
                                   (not-empty (get-in response [:headers "content-type"])))
                    {:keys [decode]} (find-encoder default-encoders content-type)]
                (update response :body decode))))})

(defn validate-response-body
  "Validate responses against the appropriate response schema.
   Optional strict mode throws an erro if it is invalid."
  ([] (validate-response-body {:strict? false}))
  ([{:keys [strict?]}]
   {:name :dci/validate-response-body
    :leave (fn [ctx] ctx)}))

(def perform-sse-request
  {:name :dci/perform-sse-request
   :leave (fn [{:keys [request params] :as ctx}]
            (assoc ctx :response #?(:clj (if (:stream params)
                                           (sse/sse-request ctx)
                                           (client/request request))
                                    :cljs {})))})

(def perform-sync-request
  {:name :dci/perform-sync-request
   :leave (fn [{:keys [request] :as ctx}]
            (assoc ctx :response #?(:clj (client/request request)
                                    :cljs {})))})

(defn ^:private process-async-response [ctx response]
  (:response (rc/context-execute (assoc ctx :response response))))

(defn ^:private process-async-error [ctx error]
  (:response (rc/context-execute (assoc ctx rc/context-error error))))

(def perform-async-request
  {:name :dci/perform-async-request
   :leave (fn [{:keys [request] :as ctx}]
            (-> ctx
              rc/remove-stack
              (assoc :response #?(:clj (client/request (assoc request :async? true)
                                         (partial process-async-response ctx)
                                         (partial process-async-error ctx))
                                  :cljs {}))))})
