(ns com.vadelabs.adapter-core.interface
  (:require
   [com.vadelabs.utils-core.interface :as uc]
   [com.wsscode.pathom.viz.ws-connector.core :as pvc]
   [com.wsscode.pathom.viz.ws-connector.pathom3 :as p.connector]
   [com.wsscode.pathom3.connect.built-in.plugins :as pbip]
   [com.wsscode.pathom3.plugin :as p.plugin]))

(defn ^:private dispatch-func
  ([_genv {:adapter/keys [type]}]
   type))

(defmulti prepare-env
  #'dispatch-func)

(defmulti initiate-env!
  #'dispatch-func)

(defmulti halt-env!
  #'dispatch-func)

(defmulti sync-schema!
  #'dispatch-func)

(defmulti pathom-env
  #'dispatch-func)

(defmulti enrich-attribute
  (fn [{:adapter/keys [type]} _attribute]
    type))

(defmulti default-actions
  (fn [{:adapter/keys [type]} _attributes]
    type))

(defn initiate!
  "Initiates all adapters and combines them into a global pathom environment"
  [{:keys [dry-run? parser] :as gopts} adapters]
  (let [adapters (if (map? adapters) [adapters] adapters)
        genv  (reduce (fn [aenv adapter]
                        (-> aenv
                          (prepare-env adapter)
                          (initiate-env! adapter)
                          (sync-schema! adapter)
                          (pathom-env adapter)))
                gopts
                adapters)
        genv (cond-> genv
               (p.plugin/register pbip/mutation-resolve-params)
               (assoc :com.wsscode.pathom3.error/lenient-mode? true ::adapters adapters)
               parser (p.connector/connect-env {::pvc/parser-id parser}))]
    genv))

(defn halt!
  "Halts the datasource adapter if possible"
  [{::keys [adapters] :as genv}]
  (for [adapter adapters]
    (halt-env! genv adapter)))

(defn ^:private attributes-map
  [attrs]
  (->> attrs
    (mapcat (fn [{:attribute/keys [local-key qualified-key] :as attr}]
              [[local-key attr]
               (when qualified-key
                 [qualified-key attr])]))
    (into {})))

(defmethod prepare-env :default
  [genv {:adapter/keys [actions attributes qualify-attributes nspace identifier type id config]
         :or {actions []}
         :as adapter}]
  (let [adapter (assoc adapter :adapter/attributes-map (attributes-map attributes))
        attributes (mapv (partial enrich-attribute adapter) attributes)
        actions (into actions (default-actions adapter attributes))]
    (cond-> genv
      (seq actions) (assoc-in [type id :actions] actions)
      (seq attributes) (assoc-in [type id :attributes] attributes)
      (seq attributes) (assoc-in [type id :attributes-map] (attributes-map attributes))
      nspace (assoc-in [type id :migration-history] (uc/prefix-keyword "." nspace :migration-history))
      nspace (assoc-in [type id :nspace] nspace)
      type (assoc-in [type id :dialect] (-> type name keyword))
      identifier (assoc-in [type id :identifier] identifier)
      (seq config) (assoc-in [type id :config] (or config {}))
      (boolean? qualify-attributes) (assoc-in [type id :qualify-attributes] qualify-attributes))))

(defmethod enrich-attribute :default
  [{:adapter/keys [qualify-attributes nspace]}
   {:attribute/keys [id local-key local-type cardinality options default doc target
                     unique primary-key remote-ns remote-entity remote-key remote-type]}]
  (let [id (or id (uc/uuid local-key))
        qualified-key (if qualify-attributes
                        (uc/keywordize nspace local-key)
                        local-key)
        qualified-target (when target
                           (if qualify-attributes
                             (uc/keywordize nspace target)
                             target))
        cardinality (or cardinality :one)]
    (cond->
      {:attribute/id id
       :attribute/local-key local-key
       :attribute/local-type local-type
       :attribute/qualified-key qualified-key
       :attribute/cardinality cardinality
       :attribute/doc (or doc "")}
      (boolean? unique) (assoc :attribute/unique unique)
      remote-ns (assoc :attribute/remote-ns remote-ns)
      remote-entity (assoc :attribute/remote-entity remote-entity)
      remote-key (assoc :attribute/remote-key remote-key)
      remote-type (assoc :attribute/remote-type remote-type)
      (seq options) (assoc :attribute/options options)
      default (assoc :attribute/default default)
      primary-key (assoc :attribute/primary-key primary-key)
      target (assoc :attribute/target target)
      qualified-target (assoc :attribute/qualified-target qualified-target))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ;; ADAPTER LIFECYCLE METHODS ;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; (defn ^:private dispatch-func
;;   ([_env {:adapter/keys [type]}]
;;    type)
;;   ([_env {:adapter/keys [type]} _attributes_or_options]
;;    type))

;; (defmulti adapter-env
;;   "Generates connection configuration based on the adapter properties for a datasource"
;;   {:arglist '([env adapter])}
;;   dispatch-func)

;; (defmulti connect
;;   "Connects to the datasource and adds connection object to the environment"
;;   {:arglist '([env adapter])}
;;   dispatch-func)

;; (defmulti disconnect
;;   "Disconnects from the datasource in the environment"
;;   {:arglist '([env adapter])}
;;   dispatch-func)

;; (defmulti actions
;;   "Get operations for the adapter to generate pathom operations"
;;   dispatch-func)

;; (defmulti schema
;;   "It converts atlas attributes to corresponsing datasource schema"
;;   {:arglist '([env adapter attributes])}
;;   dispatch-func)

;; (defmulti sync-schema
;;   "Optionally runs the migration step when it is needed for the datasource"
;;   {:arglist '([env adapter options])}
;;   dispatch-func)

;; (defmulti auto-migrate
;;   (fn [_env {:adpater/keys [type]}]
;;     type))

;; (defmulti pathom-env
;;   "Generates pathom mutations and resolvers and returns the pathom environment"
;;   {:arglist '([env adapter options])}
;;   dispatch-func)

;; (defmulti remote-attribute
;;   "Converts local to remote"
;;   (fn [{:adapter/keys [type]} _attribute]
;;     type))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ;; CORE METHODS NECESSARY FOR ADAPTER INTEGRATION ;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; (defn register
;;   ([{:keys [dry-run? parser] :as gopts} adapters]
;;    (let [adapters (if (map? adapters) [adapters] adapters)
;;          genv (->> adapters
;;                 (reduce (fn [aenv adapter]
;;                           (cond-> aenv
;;                             :always (adapter-env adapter)
;;                             ;; :always (connect adapter)
;;                             ;; :always (schema adapter)
;;                             ;; (false? dry-run?) (sync-schema adapter)
;;                             ;; :always (actions adapter)
;;                             ;; :always (pathom-env adapter)
;;                             ))

        ;;           gopts))
        ;;  genv (cond-> genv
        ;;         (p.plugin/register pbip/mutation-resolve-params)
        ;;         (assoc :com.wsscode.pathom3.error/lenient-mode? true
        ;;           ::adapters adapters)
        ;;         parser (p.connector/connect-env {::pvc/parser-id parser}))]
;;      genv)))

;; (defn deregister!
;;   "Deregisters all the connected adapters"
;;   [{::keys [adapters] :as env}]
;;   (for [adapter (keys adapters)]
;;     (disconnect env adapter)))

;; (defn ^:private nspace-exists?
;;   [k]
;;   (-> k
;;     namespace
;;     (uc/str-split ".")
;;     count
;;     (>= 2)))

;; (defn ^:private attribute
;;   [{:adapter/keys [nspace] :as adapter} attribute]
;;   (let [{:attribute/keys [id local-key local-type doc
;;                           primary-key cardinality default
;;                           options op-name inputs
;;                           outputs target params]} attribute
;;         id (or id (uc/uuid local-key))
;;         local-key (uc/keywordize nspace local-key)
;;         cardinality (or cardinality :one)
;;         inputs (mapv (fn [attribute-key]
;;                        (if (nspace-exists? attribute-key)
;;                          attribute-key
;;                          (uc/keywordize nspace attribute-key)))
;;                  inputs)
;;         outputs (mapv (fn [attribute-key]
;;                         (if (nspace-exists? attribute-key)
;;                           attribute-key
;;                           (uc/keywordize nspace attribute-key)))
;;                   outputs)]

;;     (cond-> {:attribute/id id
;;              :attribute/local-key local-key
;;              :attribute/local-type local-type
;;              :attribute/cardinality cardinality}
;;       true (merge (remote-attribute adapter attribute))
;;       (seq options) (assoc :attribute/options options)
;;       default (assoc :attribute/default default)
;;       primary-key (assoc :attribute/primary-key primary-key)
;;       doc (assoc :attribute/doc doc)
;;       op-name (assoc :attribute/op-name (uc/symbolize nspace op-name))
;;       (seq params) (assoc :attribute/params params)
;;       (seq inputs) (assoc :attribute/inputs inputs)
;;       (seq outputs) (assoc :attribute/outputs outputs)
;;       target (assoc :attribute/target (if (nspace-exists? target)
;;                                         target
;;                                         (uc/keywordize nspace target))))))

;; (defn conform-attributes
;;   ([{:adapter/keys [attributes] :as adapter}]
;;    (->> attributes
;;      (mapv (partial attribute adapter))
;;      spec/conform)))

;; (defn ^:private ref?
;;   [{:attribute/keys [local-type]}]
;;   (= local-type :attribute.type/ref))

;; (defn pathom-query
;;   ([attrs]
;;    (reduce
;;      (fn [acc {:attribute/keys [local-key target] :as attr}]
;;        (cond-> acc
;;          (ref? attr) (conj {local-key [target]})
;;          (not (ref? attr)) (conj local-key)))
;;      []
;;      attrs)))

;; (defn ^:private ident-input
;;   [attrs]
;;   (->> attrs
;;     (filter :attribute/primary-key)
;;     (mapv :attribute/local-key)))

;; (defn ^:private find-params
;;   [attrs]
;;   (->> attrs
;;     (filter :attribute/primary-key)
;;     (mapv :attribute/remote-key)))

;; (defn ^:private ident-key
;;   [attrs]
;;   (->> attrs
;;     (filter :attribute/identity?)
;;     (mapv :attribute/remote-key)
;;     first))

;; (defn ^:private operation
;;   [nspace entity attributes]
;;   (let [prefix-fn (partial uc/keywordize nspace entity)
;;         ident-key (ident-key attributes)
;;         optionals [:limit :offset :order-by]]
;;     [{:op/identifier (prefix-fn :insert)
;;       :op/type :insert
;;       :op/entity (prefix-fn)
;;       :op/params [:data]
;;       :op/output (pathom-query attributes)}
;;      #_{:op/identifier (prefix-fn :delete)
;;         :op/type :delete
;;         :op/entity (prefix-fn)
;;         :op/params [:where]
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identifier (prefix-fn :delete-many)
;;         :op/type :delete-many
;;         :op/entity (prefix-fn)
;;         :op/params [:where]
;;         :op/output (/ attributes)}
;;      #_{:op/identifier (prefix-fn :create)
;;         :op/type :create
;;         :op/entity (prefix-fn)
;;         :op/params [:data]
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identifier (prefix-fn :create-many)
;;         :op/type :create-many
;;         :op/entity (prefix-fn)
;;         :op/params [:data]
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identfier (prefix-fn :find-unique)
;;         :op/type :find-unique
;;         :op/entity (prefix-fn)
;;         :op/optionals optionals
;;         :op/input []
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identfier (prefix-fn :find-many)
;;         :op/type :find-many
;;         :op/entity (prefix-fn)
;;         :op/optionals optionals
;;         :op/input []
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identfier (prefix-fn :find-first)
;;         :op/type :find-first
;;         :op/entity (prefix-fn)
;;         :op/optionals optionals
;;         :op/input []
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identifier (prefix-fn :update)
;;         :op/type :update
;;         :op/entity (prefix-fn)
;;         :op/params [:where :data]
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identifier (prefix-fn :update-many)
;;         :op/type :update-many
;;         :op/entity (prefix-fn)
;;         :op/params [:where :data]
;;         :op/output (pathom-query attributes)}
;;      #_{:op/identifier (prefix-fn :upsert)
;;         :op/type :upsert
;;         :op/entity (prefix-fn)
;;         :op/params [:where :create :update]
;;         :op/output (pathom-query attributes)}]))

;; (defn operations
;;   [attrs]
;;   (->> attrs
;;     (uc/group-by (juxt :attribute/remote-ns :attribute/remote-entity))
;;     (reduce-kv
;;       (fn [acc [remote-ns remote-entity] attributes]
;;         (into acc (operation remote-ns remote-entity attributes)))
;;       [])))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ;; DEFAULT IMPL FOR ADAPTER LIFECYCLE METHODS ;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; (defn attributes-map
;;   [attrs]
;;   (->> attrs
;;     (mapcat (fn [{:attribute/keys [local-key] :as attr}]
;;               [[local-key attr]]))
;;     (into {})))

;; (defmethod adapter-env :default
;;   [genv {:adapter/keys [actions attributes nspace
;;                         identifier type id config]
;;          :as adapter}]

;;   (let [adapter (assoc adapter :adapter/attributes-map (attributes-map attributes))
;;         attrs (conform-attributes adapter)]
;;     (cond-> genv
;;       (seq actions) (assoc-in [type id :actions] actions)
;;       (seq attributes) (assoc-in [type id :attributes] attrs)
;;       (seq attributes) (assoc-in [type id :attributes-map] (attributes-map attrs))
;;       nspace (assoc-in [type id :schema-history] (uc/prefix-keyword "." nspace :schema-history))
;;       nspace (assoc-in [type id :nspace] nspace)
;;       type (assoc-in [type id :dialect] (-> type name keyword))
;;       identifier (assoc-in [type id :identifier] identifier)
;;       (seq config) (assoc-in [type id :config] config))))

;; (defmethod schema :default
;;   [genv {:adapter/keys [type id]} attributes]
;;   genv)

;; (defmethod disconnect :default
;;   [_ _]
;;   nil)

;; (defmethod sync-schema :default
;;   [genv _ _]
;;   genv)

;; (defmethod remote-attribute :default
;;   [{:adapter/keys [nspace]} {:attribute/keys [local-key local-type
;;                                               remote-ns remote-entity
;;                                               remote-key remote-type]}]
;;   {:attribute/remote-ns (or remote-ns nspace)
;;    :attribute/remote-entity (or remote-entity (-> local-key namespace keyword))
;;    :attribute/remote-key (or remote-key (-> local-key name keyword))
;;    :attribute/remote-type (or remote-type (-> local-type name keyword))})

;; (defn ->local-result
;;   [{:keys [attributes-map]} ast-nodes remote-result]
;;   (reduce
;;     (fn [acc {:keys [dispatch-key type]}]
;;       (let [{:attribute/keys [remote-key target local-type]} (get attributes-map dispatch-key)
;;             result-value (get remote-result remote-key)
;;             local-type (get-in attributes-map [target :attribute/local-type] local-type)
;;             result-value result-value]
;;         (cond-> acc
;;           (= type :join) (assoc-in [dispatch-key target] result-value)
;;           (= type :prop) (assoc-in [dispatch-key] result-value))))
;;     {}
;;     ast-nodes))

;; (defn ast-nodes
;;   [{:action/keys [output identifier]} penv]
;;   (let [{:keys [children] :as ast} (-> penv
;;                                      ::pcp/graph
;;                                      ::pcp/source-ast
;;                                      eql/ast->query
;;                                      identifier
;;                                      eql/query->ast)
;;         {:keys [children]} (if (seq children)
;;                              ast
;;                              (eql/query->ast output))]
;;     children))
