(ns freebie.domain
  (:require [clojure.algo.generic.functor :refer [fmap]]
            [freebie.types :refer [lift-f]]
            [freebie.util :refer [split-signatures-from-args]]
            [schema.core :as s]
            [camel-snake-kebab.core :refer [->PascalCase]]))

(defrecord DomainFnMeta
    [fn-pascal-name
     fn-kebab-name
     fn-record-ctor-name
     fn-functor-arg
     fn-args
     fn-arg-types
     fn-return-type])

(defn get-domain-fn-functor-arg [fn-body]
  (let [[sig-op sig-ty & rest] fn-body]
    (if (= sig-op :-)
      'f-next
      'next)))

(defn get-domain-fn-return-type [fn-body0]
  (let [[sig-op sig-ty & fn-body] fn-body0]
    (if (= sig-op :-)
      sig-ty
      s/Any )))

(defn get-domain-fn-metadata
  [domain-name [fn-name fn-args0 & fn-body0]]

  (when-not (symbol? fn-name)
    (throw
     (Exception.
      (str "Definitions on domain `" domain-name
           "` must start with a symbol; found `" fn-name "` instead"))))

  (when-not (vector? fn-args0)
    (throw
     (Exception.
      (str "Definition `" fn-name "` on domain `" domain-name
           "` must have arguments in a vector; found `" fn-args0
           " instead"))))

  (let [fn-pascal-name          (-> fn-name name ->PascalCase symbol)
        fn-functor-arg          (get-domain-fn-functor-arg fn-body0)
        fn-return-type          (get-domain-fn-return-type fn-body0)
        {fn-args :args
         fn-arg-types :arg-types} (split-signatures-from-args fn-args0)
        fn-record-ctor-name     (->> fn-pascal-name
                                     name
                                     (str "->")
                                     symbol)]

    {fn-name (map->DomainFnMeta {:fn-pascal-name fn-pascal-name
                                 :fn-kebab-name  fn-name
                                 :fn-record-ctor-name fn-record-ctor-name
                                 :fn-functor-arg fn-functor-arg
                                 :fn-args        fn-args
                                 :fn-arg-types   fn-arg-types
                                 :fn-return-type fn-return-type})}))

(defn mk-domain-fn-record
  [{:keys [fn-pascal-name fn-args fn-arg-types fn-functor-arg]}]
  (let [fn-args-with-types
        (reduce
         #(conj %1 %2 ':- (get fn-arg-types %2))
         []
         fn-args)

        record-args (-> fn-args-with-types
                        (concat [fn-functor-arg])
                        vec)]
    `(s/defrecord ~fn-pascal-name ~record-args)))

(defn mk-functor-multimethod
  [{:keys [fn-pascal-name fn-record-ctor-name fn-args fn-functor-arg]}]
  (let [record-sym    (gensym "record")
        fmap-f-sym    (gensym "f")]
    `(defmethod fmap ~fn-pascal-name [~fmap-f-sym ~record-sym]
       (~fn-record-ctor-name ~@(for [arg fn-args
                                     :let [arg-kw (keyword arg)]]
                                 `(~arg-kw ~record-sym))
                             ~(if (= fn-functor-arg 'next)
                                `(~fmap-f-sym (:next ~record-sym))
                                `(comp ~fmap-f-sym (:f-next ~record-sym)))))))

(defn mk-domain-helper-method
  [{:keys [fn-pascal-name
           fn-record-ctor-name
           fn-args
           fn-functor-arg
           fn-kebab-name
           fn-return-type]}]
  `(defn ~fn-kebab-name [~@fn-args]
    (let [record-value# (~fn-record-ctor-name ~@fn-args
                                              ~(if (= fn-functor-arg 'next)
                                                 nil
                                                 `(fn ~'f-next [v#]
                                                    (s/validate ~fn-return-type v#)
                                                    v#)))]
      (s/validate ~fn-pascal-name record-value#)
      (lift-f
       record-value#))))

(defn build-domain
  [domain-name domains-fn-meta]
  (let [metas (vals domains-fn-meta)]
    `(do
       ~@(mapv mk-domain-fn-record metas)
       ~@(mapv mk-functor-multimethod metas)
       ~@(mapv mk-domain-helper-method metas)
       (def ~domain-name {:domain-name ~domain-name
                          :domains-fn-meta '~domains-fn-meta}))))

(defmacro defdomain [domain-name & domain-fns]
  (when-not (symbol? domain-name)
    (throw
     (Exception.
      (str "First argument of `defdomain` must be a symbol; found instead: " domain-name))))

  (when (empty? domain-fns)
    (throw
     (Exception.
      (str "Expecting definitions on domain `" domain-name "`, but none was found."))))

  (let [domains-fn-meta (reduce #(merge %1 (get-domain-fn-metadata domain-name %2))
                                {}
                                domain-fns)]
    (build-domain domain-name
                  domains-fn-meta)))

(comment
  ;; Example
  (defdomain console-domain
    (read-input [] :- s/Str)
    (print-output [output :- s/Str])))
