(ns freebie.algebra
  (:require [clojure.algo.generic.functor :refer [fmap]]
            [clojure.string :as str]
            [freebie.types :refer [lift-f]]
            [freebie.util :refer [spine-case force-into-groups transpose]]
            [schema.core :as s]))

(defn mk-instr-field-groups
  "Group list in sub-groups of 3, composed by: instr-fn-arg-name :- fn-arg-type"
  [algebra-name instr-name instr-fn-args]
  (force-into-groups 3
                     instr-fn-args
                     (str "Expecting instruction `" instr-name
                          "` of algebra `" algebra-name
                          "` to contain attributes with type"
                          " declarations (e.g. :- Type)")))

(defn fetch-instr-return-type [instr-ctor instr-functor-arg]
  (if (= instr-functor-arg :f-next)
    (let [[_ _ _operator instr-return-type] instr-ctor]
      instr-return-type)
    nil))

(defn assert-instr-has-next-or-f-next-arg!
  [algebra-name instr-name instr-functor-arg]
  (assert (and (not (nil? instr-functor-arg))
               (#{:next :f-next} instr-functor-arg))
          (str "Expecting instruction `" instr-name
               "` of algebra `" algebra-name
               "` to contain `next` or `f-next` on it's last attribute")))

(defn assert-instr-has-input-type-declarations!
  [{:keys [algebra-name instr-name instr-fn-args-groups]}]
  (let [args-without-signature
        (->> instr-fn-args-groups
             (filter (fn [[_ operator _]]
                       (not= operator ':-)))
             (mapv first))

        any-args-without-type?
        (not (empty? args-without-signature))]

    (assert (not any-args-without-type?)
            (str "Expecting instruction `" instr-name
                 "` of algebra `" algebra-name "` to have fields with types,"
                 " fields with missing types are`"
                 (str/join "`, `" args-without-signature "`")))))

(defn assert-instr-has-output-type-declaration!
  [{:keys [algebra-name instr-ctor instr-name instr-functor-arg instr-return-type]}]
  (when (= instr-functor-arg :f-next)
    (let [[_ _ operator _ & rest] instr-ctor]
      (assert (= operator ':-)
              (str "Expecting instruction `" instr-name
                   "` of algebra `" algebra-name
                   "` to have a return type declaration (e.g. :- Type);"
                   " this is required for instructions with an `f-next`"
                   " on their last arg")))))


(defn mk-algebra-instr-env
  [{:keys [algebra-name] :as env0} instr-ctor]
  (let [instr-name           (first instr-ctor)
        instr-functor-arg    (-> instr-ctor second last keyword)

        _
        (assert-instr-has-next-or-f-next-arg! algebra-name instr-name instr-functor-arg)

        instr-fn-args        (-> instr-ctor second butlast)
        instr-fn-args-groups (mk-instr-field-groups algebra-name instr-name instr-fn-args)

        env
        (assoc env0
               :instr-ctor           instr-ctor
               :instr-name           instr-name
               :instr-functor-arg    instr-functor-arg
               :instr-fn-args        instr-fn-args
               :instr-fn-args-groups instr-fn-args-groups
               :instr-fn-args-names  (mapv first instr-fn-args-groups)
               :instr-return-type    (fetch-instr-return-type instr-ctor instr-functor-arg))]
    ;; validations
    (assert-instr-has-input-type-declarations! env)
    (assert-instr-has-output-type-declaration! env)
    env))

(defn mk-algebra-util-fn
  [{:keys [algebra-name
           instr-ctor
           instr-name
           instr-functor-arg
           instr-return-type
           instr-fn-args
           instr-fn-args-groups
           instr-fn-args-names]
    :as env}]
  (let [ctor-name          (->> instr-name name (str "->") symbol)
        spine-record-name  (-> instr-name name spine-case symbol)]

    `(s/defn ~spine-record-name
       ([~@instr-fn-args]
        (let [record-value#
              (~ctor-name ~@instr-fn-args-names
                          ~(if (= :next instr-functor-arg)
                             nil
                             `(fn ~(symbol (str spine-record-name "-f-next"))  [val#]
                                (s/validate ~instr-return-type val#)
                                val#)))]
          (s/validate ~instr-name record-value#)
          (lift-f record-value#))))))

(defn mk-algebra-ctor-functor
  [{:keys [algebra-name instr-ctor instr-fn-args-names instr-functor-arg]}]
  (let [ctor-name        (first instr-ctor)
        ctor-fn-name     (symbol (str  "->" (name ctor-name)))
        normal-args-kw   (mapv keyword instr-fn-args-names)

        fn-var (gensym "f")
        instance-var (gensym "instance")]
    `(defmethod fmap ~ctor-name [~fn-var ~instance-var]
       (~ctor-fn-name  ~@(for [arg-kw normal-args-kw]
                           `(~arg-kw ~instance-var))
                       ~(if (= instr-functor-arg :next)
                          `(~fn-var (:next ~instance-var))
                          ;; else :f-next
                          `(comp ~fn-var (:f-next ~instance-var)))))))

(defn mk-algebra-map-entry
  [{:keys [instr-name instr-functor-arg]}]
  {instr-name {:functor-arg instr-functor-arg}})

(defn strip-return-type-decl-from-instr
  [{:keys [instr-ctor instr-functor-arg]}]
  (if (= instr-functor-arg :f-next)
    (let [[instr-name instr-record-args _ _ & rest] instr-ctor]
      (concat [instr-name instr-record-args] rest))
    ;; else
    instr-ctor))

(defn mk-algebra-ctor-record
  [env]
  (let [instr-ctor (strip-return-type-decl-from-instr env)]
    `(s/defrecord ~@instr-ctor)))



(defn defalgebra* [algebra-name algebra-instr-ctors]
  (let [env
        {:algebra-name algebra-name
         :algebra-instr-ctors algebra-instr-ctors}

        apply-to-each-instr-ctor
        (fn each-instr-ctor [fs]
          (transpose
           (for [instr-ctor algebra-instr-ctors
                 :let [env1 (mk-algebra-instr-env env instr-ctor)
                       f   (apply juxt fs)]]
             (f env1))))

        ctor-names (mapv first algebra-instr-ctors)

        [records functors util-fns algebra-map-entries]
        (apply-to-each-instr-ctor [mk-algebra-ctor-record
                                   mk-algebra-ctor-functor
                                   mk-algebra-util-fn
                                   mk-algebra-map-entry])]
    `(do
       ~@records
       ~@functors
       ~@util-fns
       (def ~algebra-name ~(apply merge algebra-map-entries)))))

(defmacro defalgebra [algebra-name & algebra-instr-ctors]
  (defalgebra* algebra-name algebra-instr-ctors))
