(ns clj-hazelcast.mr
  (:import (com.hazelcast.mapreduce Mapper Reducer ReducerFactory KeyValueSource Combiner CombinerFactory Collator)
           (com.hazelcast.core ExecutionCallback)
           (cljhazelcast.remote RemoteMapper RemoteReducerFactory RemoteCombinerFactory RemoteCollator))
  (:require [clj-hazelcast.core :as hazelcast]
            [clojure.tools.logging :as log]))

(defn emit-mapped-pairs [pairs collector]
  (doall
    (map #(do
           (let [emit-key (first %1)
                 emit-value (second %1)]
             (.emit collector emit-key emit-value)))
         pairs)))

;TODO issues for mapper and reducer macro
;difficult to test
;not binding
(defmacro defmapper
  "runs the function f over the content

  f is a function of two arguments, key and value.

  f must return a sequence of *pairs* like
   [[key1 value1] [key2 value2] ...]
  "
  [fname args & body]
  `(let [instance# (RemoteMapper.
                     (pr-str '(fn ~args ~@body)))]
     (def ~fname instance#)))

(defmacro defreducer
  "runs the reducer function rf over the content

  rf is a function of three arguments,

  key,value and acc

  rf should return the accumulated result.
  "
  [fname args & body]
  `(let [instance# (RemoteReducerFactory.
                     (pr-str '(fn ~args ~@body)))]
     (def ~fname instance#)))


(defmacro defcombiner
  "runs the combiner function cf over the content

  cf is a function of three arguments,

  key,value and acc

  cf should set the val eventually
  "
  [fname args & body]
  `(let [instance# (RemoteCombinerFactory.
                     (pr-str '(fn ~args ~@body)))]
     (def ~fname instance#)))

(defmacro defcollator
  "
  runs the collactor function f over the content

  f is a function of one argument, a sequence of Entry objects,
  where Entry's are generated by the reducers output:

  [Entry1 Entry2 ...]
  "
  [fname args & body]
  `(let [instance# (RemoteCollator.
                     (pr-str '(fn ~args ~@body)))]
     (def ~fname instance#)))

(defn make-job-tracker [hz-instance]
  (.getJobTracker hz-instance "default"))

(defn submit-job "returns the future object from Hazelcast Mapreduce Api"
  [{:keys [map mapper-fn combiner-fn combiner-acc reducer-fn reducer-acc collator-fn tracker]}]
  (let [src (KeyValueSource/fromMap map)
        ;mapper (hmap mapper-fn)
        mapper mapper-fn
        ;reducer (hreducefactory reducer-fn reducer-acc)
        reducer reducer-fn
        job (atom (-> (.newJob tracker src)
                      (.mapper mapper)))]
    (log/debugf "Submitting new MR Job ...")
    (if-not (nil? combiner-fn)
      (do
        (log/debugf "No Combiner defined for job...")
        (reset! job (-> @job
                        (.combiner combiner-fn))))
      )
    (if-not (nil? collator-fn)
      (do
        (log/debugf "No Collator defined for job...")
        (reset! job (-> @job
                        (.reducer reducer)
                        (.submit collator-fn))))
      (reset! job (-> @job
                      (.reducer reducer)
                      (.submit))))

    (.andThen @job (proxy [ExecutionCallback] []
                     (onResponse [res]
                       (log/debugf "Calculation finished %s" res))
                     (onFailure [t]
                       (.printStackTrace t))))
    (log/debugf "Callback registered ...")
    @job))

